// Test script to verify all report CRUD operations are working properly
import fetch from "node-fetch"

const BASE_URL = "http://localhost:3000/api"

// Test data for different report types
const testSalesData = {
  date: new Date().toISOString().split("T")[0],
  branchId: "507f1f77bcf86cd799439011", // Sample ObjectId
  branchName: "Test Branch",
  cashInShop: 500,
  cashDelivery: 300,
  cardInShop: 400,
  cardDelivery: 200,
  totalCash: 800,
  totalCard: 600,
  totalSales: 1400,
  notes: "Test sales record",
}

const testExpenseData = {
  date: new Date().toISOString().split("T")[0],
  branchId: "507f1f77bcf86cd799439011",
  branchName: "Test Branch",
  category: "managing-expenses",
  subcategory: "office-supplies",
  amount: 150,
  description: "Test expense record",
  paymentMethod: "cash",
  fakturaNo: "TEST-001",
  notes: "Test expense",
}

const testInventoryData = {
  date: new Date().toISOString().split("T")[0],
  branchId: "507f1f77bcf86cd799439011",
  branchName: "Test Branch",
  itemName: "Test Item",
  category: "daily-consumption",
  quantity: 10,
  unit: "kg",
  unitPrice: 25,
  totalValue: 250,
  notes: "Test inventory record",
}

async function testCRUDOperations(endpoint, testData, entityName) {
  console.log(`\n=== Testing ${entityName} CRUD Operations ===`)

  try {
    // Test CREATE
    console.log(`[v0] Testing CREATE ${entityName}...`)
    const createResponse = await fetch(`${BASE_URL}/${endpoint}`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(testData),
    })

    if (!createResponse.ok) {
      console.log(`❌ CREATE ${entityName} failed: ${createResponse.status}`)
      return false
    }

    const createResult = await createResponse.json()
    console.log(`✅ CREATE ${entityName} successful`)
    const createdId = createResult.data?._id || createResult._id

    if (!createdId) {
      console.log(`❌ No ID returned from CREATE ${entityName}`)
      return false
    }

    // Test READ (GET all)
    console.log(`[v0] Testing READ ${entityName} (all)...`)
    const readAllResponse = await fetch(`${BASE_URL}/${endpoint}`)

    if (!readAllResponse.ok) {
      console.log(`❌ READ ${entityName} (all) failed: ${readAllResponse.status}`)
      return false
    }

    const readAllResult = await readAllResponse.json()
    console.log(`✅ READ ${entityName} (all) successful - Found ${readAllResult.data?.length || 0} records`)

    // Test READ (GET by ID)
    console.log(`[v0] Testing READ ${entityName} (by ID)...`)
    const readByIdResponse = await fetch(`${BASE_URL}/${endpoint}/${createdId}`)

    if (!readByIdResponse.ok) {
      console.log(`❌ READ ${entityName} (by ID) failed: ${readByIdResponse.status}`)
      return false
    }

    const readByIdResult = await readByIdResponse.json()
    console.log(`✅ READ ${entityName} (by ID) successful`)

    // Test UPDATE
    console.log(`[v0] Testing UPDATE ${entityName}...`)
    const updateData = { ...testData, notes: `Updated test ${entityName.toLowerCase()}` }
    const updateResponse = await fetch(`${BASE_URL}/${endpoint}/${createdId}`, {
      method: "PUT",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(updateData),
    })

    if (!updateResponse.ok) {
      console.log(`❌ UPDATE ${entityName} failed: ${updateResponse.status}`)
      return false
    }

    console.log(`✅ UPDATE ${entityName} successful`)

    // Test DELETE
    console.log(`[v0] Testing DELETE ${entityName}...`)
    const deleteResponse = await fetch(`${BASE_URL}/${endpoint}/${createdId}`, {
      method: "DELETE",
    })

    if (!deleteResponse.ok) {
      console.log(`❌ DELETE ${entityName} failed: ${deleteResponse.status}`)
      return false
    }

    console.log(`✅ DELETE ${entityName} successful`)

    return true
  } catch (error) {
    console.log(`❌ Error testing ${entityName} CRUD operations:`, error.message)
    return false
  }
}

async function testReportFiltering() {
  console.log("\n=== Testing Report Filtering ===")

  try {
    // Test sales filtering by date range
    console.log("[v0] Testing sales filtering by date range...")
    const fromDate = "2024-01-01"
    const toDate = "2024-12-31"
    const salesFilterResponse = await fetch(`${BASE_URL}/sales?fromDate=${fromDate}&toDate=${toDate}`)

    if (salesFilterResponse.ok) {
      const salesFilterResult = await salesFilterResponse.json()
      console.log(`✅ Sales date filtering successful - Found ${salesFilterResult.data?.length || 0} records`)
    } else {
      console.log("❌ Sales date filtering failed")
    }

    // Test expense filtering by category
    console.log("[v0] Testing expense filtering by category...")
    const expenseFilterResponse = await fetch(`${BASE_URL}/expenses?category=managing-expenses`)

    if (expenseFilterResponse.ok) {
      const expenseFilterResult = await expenseFilterResponse.json()
      console.log(`✅ Expense category filtering successful - Found ${expenseFilterResult.data?.length || 0} records`)
    } else {
      console.log("❌ Expense category filtering failed")
    }

    // Test branch filtering
    console.log("[v0] Testing branch filtering...")
    const branchFilterResponse = await fetch(`${BASE_URL}/sales?branchId=507f1f77bcf86cd799439011`)

    if (branchFilterResponse.ok) {
      const branchFilterResult = await branchFilterResponse.json()
      console.log(`✅ Branch filtering successful - Found ${branchFilterResult.data?.length || 0} records`)
    } else {
      console.log("❌ Branch filtering failed")
    }
  } catch (error) {
    console.log("❌ Error testing report filtering:", error.message)
  }
}

async function testReportGeneration() {
  console.log("\n=== Testing Report Generation ===")

  try {
    // Test dashboard metrics
    console.log("[v0] Testing dashboard metrics...")
    const metricsResponse = await fetch(`${BASE_URL}/dashboard/metrics`)

    if (metricsResponse.ok) {
      const metricsResult = await metricsResponse.json()
      console.log("✅ Dashboard metrics generation successful")
      console.log(`   - Total Sales: ${metricsResult.totalSales || 0}`)
      console.log(`   - Total Expenses: ${metricsResult.totalExpenses || 0}`)
      console.log(`   - Active Branches: ${metricsResult.activeBranches || 0}`)
    } else {
      console.log("❌ Dashboard metrics generation failed")
    }
  } catch (error) {
    console.log("❌ Error testing report generation:", error.message)
  }
}

async function runAllTests() {
  console.log("🚀 Starting comprehensive report function testing...\n")

  const testResults = []

  // Test all CRUD operations
  testResults.push(await testCRUDOperations("sales", testSalesData, "Sales"))
  testResults.push(await testCRUDOperations("expenses", testExpenseData, "Expenses"))
  testResults.push(await testCRUDOperations("inventory", testInventoryData, "Inventory"))

  // Test filtering and report generation
  await testReportFiltering()
  await testReportGeneration()

  // Summary
  const passedTests = testResults.filter((result) => result).length
  const totalTests = testResults.length

  console.log("\n=== Test Summary ===")
  console.log(`✅ Passed: ${passedTests}/${totalTests} CRUD test suites`)
  console.log(`${passedTests === totalTests ? "🎉 All tests passed!" : "⚠️  Some tests failed"}`)

  if (passedTests === totalTests) {
    console.log("\n📊 Report Functions Status:")
    console.log("✅ Sales Report CRUD - Fully functional")
    console.log("✅ Expense Report CRUD - Fully functional")
    console.log("✅ Inventory Report CRUD - Fully functional")
    console.log("✅ Report Filtering - Working")
    console.log("✅ Dashboard Metrics - Working")
    console.log("\n🎯 All report functions are ready for production use!")
  }
}

// Run the tests
runAllTests().catch(console.error)
