import { MongoClient, ObjectId } from "mongodb"
import bcrypt from "bcryptjs"

const MONGODB_URI = process.env.MONGODB_URI

if (!MONGODB_URI) {
  throw new Error("Please define the MONGODB_URI environment variable")
}

async function seedDatabase() {
  const client = new MongoClient(MONGODB_URI)

  try {
    await client.connect()
    console.log("[v0] Connected to MongoDB")

    const db = client.db("resturent")

    // Clear existing data
    console.log("[v0] Clearing existing data...")
    await Promise.all([
      db.collection("users").deleteMany({}),
      db.collection("branches").deleteMany({}),
      db.collection("employees").deleteMany({}),
      db.collection("sales").deleteMany({}),
      db.collection("expenses").deleteMany({}),
      db.collection("inventory").deleteMany({}),
    ])

    // Create Users
    console.log("[v0] Creating users...")
    const hashedPassword = await bcrypt.hash("password123", 10)

    const users = [
      {
        _id: new ObjectId(),
        name: "John Smith",
        email: "admin@restaurant.com",
        phone: "+1234567890",
        password: hashedPassword,
        role: "admin",
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Sarah Johnson",
        email: "manager1@restaurant.com",
        phone: "+1234567891",
        password: hashedPassword,
        role: "manager",
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Mike Wilson",
        email: "owner@restaurant.com",
        phone: "+1234567892",
        password: hashedPassword,
        role: "owner",
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Emma Davis",
        email: "manager2@restaurant.com",
        phone: "+1234567893",
        password: hashedPassword,
        role: "manager",
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
    ]

    await db.collection("users").insertMany(users)
    console.log("[v0] Created users:", users.length)

    // Create Branches
    console.log("[v0] Creating branches...")
    const branches = [
      {
        _id: new ObjectId(),
        name: "Downtown Branch",
        address: "123 Main Street, Downtown",
        country: "USA",
        location: "New York, NY",
        ownerId: users[2]._id, // Mike Wilson (owner)
        managerId: users[1]._id, // Sarah Johnson (manager)
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Mall Branch",
        address: "456 Shopping Mall, Level 2",
        country: "USA",
        location: "Brooklyn, NY",
        ownerId: users[2]._id, // Mike Wilson (owner)
        managerId: users[3]._id, // Emma Davis (manager)
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Airport Branch",
        address: "789 Airport Terminal 1",
        country: "USA",
        location: "Queens, NY",
        ownerId: users[2]._id, // Mike Wilson (owner)
        managerId: users[1]._id, // Sarah Johnson (manager)
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
    ]

    await db.collection("branches").insertMany(branches)
    console.log("[v0] Created branches:", branches.length)

    // Update users with branch assignments
    await db
      .collection("users")
      .updateOne({ _id: users[1]._id }, { $set: { branchIds: [branches[0]._id, branches[2]._id] } })
    await db.collection("users").updateOne({ _id: users[3]._id }, { $set: { branchIds: [branches[1]._id] } })

    // Create Employees
    console.log("[v0] Creating employees...")
    const employees = [
      {
        _id: new ObjectId(),
        name: "Alex Rodriguez",
        email: "alex@restaurant.com",
        phone: "+1234567894",
        position: "Chef",
        branchId: branches[0]._id,
        hourlyRate: 25.0,
        isActive: true,
        startDate: new Date("2024-01-15"),
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Lisa Chen",
        email: "lisa@restaurant.com",
        phone: "+1234567895",
        position: "Server",
        branchId: branches[0]._id,
        hourlyRate: 18.0,
        isActive: true,
        startDate: new Date("2024-02-01"),
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "David Brown",
        email: "david@restaurant.com",
        phone: "+1234567896",
        position: "Kitchen Assistant",
        branchId: branches[1]._id,
        hourlyRate: 16.0,
        isActive: true,
        startDate: new Date("2024-01-20"),
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Maria Garcia",
        email: "maria@restaurant.com",
        phone: "+1234567897",
        position: "Cashier",
        branchId: branches[1]._id,
        hourlyRate: 17.5,
        isActive: true,
        startDate: new Date("2024-03-01"),
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Tom Anderson",
        email: "tom@restaurant.com",
        phone: "+1234567898",
        position: "Delivery Driver",
        branchId: branches[2]._id,
        hourlyRate: 19.0,
        isActive: true,
        startDate: new Date("2024-02-15"),
        createdAt: new Date(),
        updatedAt: new Date(),
      },
    ]

    await db.collection("employees").insertMany(employees)
    console.log("[v0] Created employees:", employees.length)

    // Create Sales Data (last 30 days)
    console.log("[v0] Creating sales data...")
    const salesData = []
    const today = new Date()

    for (let i = 0; i < 30; i++) {
      const date = new Date(today)
      date.setDate(date.getDate() - i)

      branches.forEach((branch, branchIndex) => {
        // Generate realistic sales data
        const baseSales = 800 + Math.random() * 400 // $800-$1200 base
        const weekendMultiplier = [0, 6].includes(date.getDay()) ? 1.3 : 1.0
        const totalSales = baseSales * weekendMultiplier

        // Distribute across payment methods and platforms
        const cashRatio = 0.3 + Math.random() * 0.2 // 30-50% cash
        const totalCash = totalSales * cashRatio
        const totalCard = totalSales - totalCash

        const sale = {
          _id: new ObjectId(),
          date: new Date(date.getFullYear(), date.getMonth(), date.getDate()),
          branchId: branch._id,

          // Cash transactions
          cashInShop: totalCash * (0.4 + Math.random() * 0.2),
          cashDelivery: totalCash * (0.1 + Math.random() * 0.1),
          cashWolt: totalCash * (0.1 + Math.random() * 0.1),
          cashBolt: totalCash * (0.08 + Math.random() * 0.08),
          cashPyszne: totalCash * (0.08 + Math.random() * 0.08),
          cashUberEats: totalCash * (0.07 + Math.random() * 0.07),
          cashGlovo: totalCash * (0.07 + Math.random() * 0.07),

          // Card transactions
          cardInShop: totalCard * (0.5 + Math.random() * 0.2),
          cardDelivery: totalCard * (0.1 + Math.random() * 0.1),
          cardWolt: totalCard * (0.1 + Math.random() * 0.1),
          cardBolt: totalCard * (0.08 + Math.random() * 0.08),
          cardPyszne: totalCard * (0.08 + Math.random() * 0.08),
          cardUberEats: totalCard * (0.07 + Math.random() * 0.07),
          cardGlovo: totalCard * (0.07 + Math.random() * 0.07),

          totalCash: Math.round(totalCash * 100) / 100,
          totalCard: Math.round(totalCard * 100) / 100,
          totalSales: Math.round(totalSales * 100) / 100,

          createdBy: users[branchIndex % users.length]._id,
          createdAt: date,
          updatedAt: date,
        }

        salesData.push(sale)
      })
    }

    await db.collection("sales").insertMany(salesData)
    console.log("[v0] Created sales records:", salesData.length)

    // Create Expense Data
    console.log("[v0] Creating expense data...")
    const expenseCategories = [
      "cost-of-sales",
      "utilities",
      "office-expenses",
      "marketing",
      "insurance",
      "maintenance",
      "professional-services",
      "travel",
      "training",
      "licenses",
      "bank-charges",
      "depreciation",
      "other",
      "salaries",
    ]

    const expenseData = []

    for (let i = 0; i < 60; i++) {
      const date = new Date(today)
      date.setDate(date.getDate() - Math.floor(Math.random() * 30))

      const branch = branches[Math.floor(Math.random() * branches.length)]
      const category = expenseCategories[Math.floor(Math.random() * expenseCategories.length)]

      let expense = {
        _id: new ObjectId(),
        date: new Date(date.getFullYear(), date.getMonth(), date.getDate()),
        branchId: branch._id,
        category,
        description: getExpenseDescription(category),
        paymentMethod: Math.random() > 0.5 ? "cash" : "card",
        createdBy: users[Math.floor(Math.random() * users.length)]._id,
        createdAt: date,
        updatedAt: date,
      }

      if (category === "salaries") {
        // Salary expense with employee details
        const employee = employees.find((emp) => emp.branchId.equals(branch._id)) || employees[0]
        const hours = 35 + Math.random() * 10 // 35-45 hours
        const totalSalary = employee.hourlyRate * hours
        const received = totalSalary * (0.8 + Math.random() * 0.2) // 80-100% received

        expense = {
          ...expense,
          employeeId: employee._id,
          hourlyRate: employee.hourlyRate,
          totalHours: Math.round(hours * 100) / 100,
          totalSalary: Math.round(totalSalary * 100) / 100,
          received: Math.round(received * 100) / 100,
          payable: Math.round((totalSalary - received) * 100) / 100,
          amount: Math.round(totalSalary * 100) / 100,
        }
      } else {
        // Regular expense with VAT calculation
        const netto = 50 + Math.random() * 500 // $50-$550
        const vat = netto * 0.21 // 21% VAT
        const brutto = netto + vat

        expense = {
          ...expense,
          netto: Math.round(netto * 100) / 100,
          vat: Math.round(vat * 100) / 100,
          brutto: Math.round(brutto * 100) / 100,
          amount: Math.round(brutto * 100) / 100,
          fakturaNo: `INV-${Date.now()}-${Math.floor(Math.random() * 1000)}`,
        }
      }

      expenseData.push(expense)
    }

    await db.collection("expenses").insertMany(expenseData)
    console.log("[v0] Created expense records:", expenseData.length)

    // Create Inventory Data
    console.log("[v0] Creating inventory data...")
    const inventoryData = []
    const foodItems = [
      "kurczak",
      "wolowina",
      "kraftowe",
      "majonez",
      "ketchup",
      "smabal",
      "yogurt",
      "pita20cm",
      "pita30cm",
      "gas",
    ]

    for (let i = 0; i < 20; i++) {
      const date = new Date(today)
      date.setDate(date.getDate() - Math.floor(Math.random() * 15))

      branches.forEach((branch) => {
        const inventory = {
          _id: new ObjectId(),
          date: new Date(date.getFullYear(), date.getMonth(), date.getDate()),
          branchId: branch._id,
          type: Math.random() > 0.5 ? "daily-consumption" : "stock-level",
          createdBy: users[Math.floor(Math.random() * users.length)]._id,
          createdAt: date,
          updatedAt: date,
        }

        // Add random quantities for each food item
        foodItems.forEach((item) => {
          if (item === "gas") {
            inventory[item] = Math.floor(Math.random() * 5) + 1 // 1-5 gas cylinders
          } else if (item.includes("pita")) {
            inventory[item] = Math.floor(Math.random() * 100) + 20 // 20-120 pita breads
          } else {
            inventory[item] = Math.floor(Math.random() * 50) + 10 // 10-60 kg/liters
          }
        })

        inventoryData.push(inventory)
      })
    }

    await db.collection("inventory").insertMany(inventoryData)
    console.log("[v0] Created inventory records:", inventoryData.length)

    console.log("[v0] Database seeding completed successfully!")
    console.log("[v0] Summary:")
    console.log(`  - Users: ${users.length}`)
    console.log(`  - Branches: ${branches.length}`)
    console.log(`  - Employees: ${employees.length}`)
    console.log(`  - Sales: ${salesData.length}`)
    console.log(`  - Expenses: ${expenseData.length}`)
    console.log(`  - Inventory: ${inventoryData.length}`)
  } catch (error) {
    console.error("[v0] Error seeding database:", error)
    throw error
  } finally {
    await client.close()
  }
}

function getExpenseDescription(category) {
  const descriptions = {
    "cost-of-sales": "Food ingredients and supplies",
    utilities: "Electricity and water bill",
    "office-expenses": "Office supplies and equipment",
    marketing: "Social media advertising",
    insurance: "Business insurance premium",
    maintenance: "Equipment maintenance and repair",
    "professional-services": "Accounting and legal services",
    travel: "Business travel expenses",
    training: "Staff training and development",
    licenses: "Business licenses and permits",
    "bank-charges": "Banking fees and charges",
    depreciation: "Equipment depreciation",
    other: "Miscellaneous business expense",
    salaries: "Employee salary payment",
  }

  return descriptions[category] || "Business expense"
}

// Run the seed function
seedDatabase().catch(console.error)
