import { getDatabase } from "../lib/mongodb"
import { ObjectId } from "mongodb"
import bcrypt from "bcryptjs"

async function seedDatabase() {
  console.log("[v0] Starting database seeding...")

  try {
    const db = await getDatabase()

    // Clear existing data
    await db.collection("users").deleteMany({})
    await db.collection("branches").deleteMany({})
    await db.collection("sales").deleteMany({})
    await db.collection("expenses").deleteMany({})
    await db.collection("inventory").deleteMany({})
    await db.collection("employees").deleteMany({})

    console.log("[v0] Cleared existing collections")

    // Create branches first
    const branches = [
      {
        _id: new ObjectId(),
        name: "Main Branch",
        address: "123 Main St, City Center",
        phone: "+1-555-0101",
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Downtown Branch",
        address: "456 Downtown Ave, Business District",
        phone: "+1-555-0102",
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Mall Branch",
        address: "789 Shopping Mall, Level 2",
        phone: "+1-555-0103",
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
    ]

    await db.collection("branches").insertMany(branches)
    console.log("[v0] Created branches")

    // Create users with different roles
    const hashedPassword = await bcrypt.hash("password123", 10)
    const users = [
      {
        _id: new ObjectId(),
        name: "John Admin",
        email: "admin@company.com",
        phone: "+1-555-1001",
        password: hashedPassword,
        role: "admin",
        branchIds: branches.map((b) => b._id),
        createdAt: new Date(),
        updatedAt: new Date(),
        isActive: true,
      },
      {
        _id: new ObjectId(),
        name: "Sarah Manager",
        email: "manager@company.com",
        phone: "+1-555-1002",
        password: hashedPassword,
        role: "manager",
        branchIds: [branches[0]._id],
        createdAt: new Date(),
        updatedAt: new Date(),
        isActive: true,
      },
      {
        _id: new ObjectId(),
        name: "Mike Owner",
        email: "owner@company.com",
        phone: "+1-555-1003",
        password: hashedPassword,
        role: "owner",
        branchIds: branches.map((b) => b._id),
        createdAt: new Date(),
        updatedAt: new Date(),
        isActive: true,
      },
    ]

    await db.collection("users").insertMany(users)
    console.log("[v0] Created users")

    // Create employees
    const employees = [
      {
        _id: new ObjectId(),
        name: "Alice Johnson",
        email: "alice@company.com",
        phone: "+1-555-2001",
        branchId: branches[0]._id,
        position: "Cashier",
        salary: 35000,
        hireDate: new Date("2024-01-15"),
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Bob Smith",
        email: "bob@company.com",
        phone: "+1-555-2002",
        branchId: branches[1]._id,
        position: "Sales Associate",
        salary: 32000,
        hireDate: new Date("2024-02-01"),
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
      {
        _id: new ObjectId(),
        name: "Carol Davis",
        email: "carol@company.com",
        phone: "+1-555-2003",
        branchId: branches[2]._id,
        position: "Supervisor",
        salary: 42000,
        hireDate: new Date("2023-11-10"),
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      },
    ]

    await db.collection("employees").insertMany(employees)
    console.log("[v0] Created employees")

    // Create sample sales data
    const salesData = []
    const currentDate = new Date()

    for (let i = 0; i < 50; i++) {
      const randomBranch = branches[Math.floor(Math.random() * branches.length)]
      const randomUser = users[Math.floor(Math.random() * users.length)]
      const randomDate = new Date(currentDate.getTime() - Math.random() * 90 * 24 * 60 * 60 * 1000) // Last 90 days

      salesData.push({
        _id: new ObjectId(),
        branchId: randomBranch._id,
        userId: randomUser._id,
        items: [
          {
            name: `Product ${i + 1}`,
            quantity: Math.floor(Math.random() * 5) + 1,
            unitPrice: Math.floor(Math.random() * 100) + 10,
            total: 0,
          },
        ],
        totalAmount: Math.floor(Math.random() * 500) + 50,
        date: randomDate,
        createdAt: randomDate,
      })
    }

    await db.collection("sales").insertMany(salesData)
    console.log("[v0] Created sales data")

    // Create sample expense data
    const expenseCategories = ["Rent", "Utilities", "Supplies", "Marketing", "Maintenance", "Staff"]
    const expenseData = []

    for (let i = 0; i < 30; i++) {
      const randomBranch = branches[Math.floor(Math.random() * branches.length)]
      const randomUser = users[Math.floor(Math.random() * users.length)]
      const randomCategory = expenseCategories[Math.floor(Math.random() * expenseCategories.length)]
      const randomDate = new Date(currentDate.getTime() - Math.random() * 90 * 24 * 60 * 60 * 1000)

      expenseData.push({
        _id: new ObjectId(),
        branchId: randomBranch._id,
        userId: randomUser._id,
        category: randomCategory,
        description: `${randomCategory} expense for ${randomBranch.name}`,
        amount: Math.floor(Math.random() * 1000) + 100,
        date: randomDate,
        createdAt: randomDate,
      })
    }

    await db.collection("expenses").insertMany(expenseData)
    console.log("[v0] Created expense data")

    // Create sample inventory data
    const inventoryItems = ["Coffee Beans", "Milk", "Sugar", "Cups", "Napkins", "Pastries", "Tea", "Syrup"]
    const inventoryData = []

    branches.forEach((branch) => {
      inventoryItems.forEach((item) => {
        inventoryData.push({
          _id: new ObjectId(),
          branchId: branch._id,
          itemName: item,
          quantity: Math.floor(Math.random() * 100) + 10,
          unitPrice: Math.floor(Math.random() * 20) + 5,
          category: "Food & Beverage",
          lastUpdated: new Date(),
          createdAt: new Date(),
        })
      })
    })

    await db.collection("inventory").insertMany(inventoryData)
    console.log("[v0] Created inventory data")

    console.log("[v0] Database seeding completed successfully!")
    console.log(`[v0] Created:`)
    console.log(`[v0] - ${branches.length} branches`)
    console.log(`[v0] - ${users.length} users`)
    console.log(`[v0] - ${employees.length} employees`)
    console.log(`[v0] - ${salesData.length} sales records`)
    console.log(`[v0] - ${expenseData.length} expense records`)
    console.log(`[v0] - ${inventoryData.length} inventory items`)
  } catch (error) {
    console.error("[v0] Error seeding database:", error)
    throw error
  }
}

// Run the seeding function
seedDatabase().catch(console.error)
