import { Server as SocketIOServer } from "socket.io"
import type { Server as HTTPServer } from "http"
import { verifySession } from "@/lib/auth/session"
import type { ServerToClientEvents, ClientToServerEvents, InterServerEvents, SocketData } from "./types"

let io: SocketIOServer<ClientToServerEvents, ServerToClientEvents, InterServerEvents, SocketData>

export const initializeWebSocket = (server: HTTPServer) => {
  io = new SocketIOServer<ClientToServerEvents, ServerToClientEvents, InterServerEvents, SocketData>(server, {
    cors: {
      origin: process.env.NODE_ENV === "production" ? false : ["http://localhost:3000"],
      methods: ["GET", "POST"],
    },
  })

  io.use(async (socket, next) => {
    try {
      const token = socket.handshake.auth.token
      if (!token) {
        return next(new Error("Authentication error"))
      }

      // Verify session token
      const session = await verifySession()
      if (!session) {
        return next(new Error("Invalid session"))
      }

      socket.data = {
        userId: session.userId,
        email: session.email,
        role: session.role,
      }

      next()
    } catch (error) {
      next(new Error("Authentication error"))
    }
  })

  io.on("connection", (socket) => {
    console.log(`User ${socket.data.email} connected`)

    // Join user to their role-based room
    socket.join(`role:${socket.data.role}`)
    socket.join(`user:${socket.data.userId}`)

    socket.on("join-room", (room) => {
      socket.join(room)
      console.log(`User ${socket.data.email} joined room: ${room}`)
    })

    socket.on("leave-room", (room) => {
      socket.leave(room)
      console.log(`User ${socket.data.email} left room: ${room}`)
    })

    socket.on("ping", () => {
      socket.emit("notification", {
        type: "info",
        title: "Connection",
        message: "Pong! Connection is active",
        timestamp: new Date().toISOString(),
      })
    })

    socket.on("disconnect", () => {
      console.log(`User ${socket.data.email} disconnected`)
    })
  })

  return io
}

export const getIO = () => {
  if (!io) {
    throw new Error("Socket.io not initialized")
  }
  return io
}

// Utility functions for broadcasting updates
export const broadcastDashboardUpdate = (type: "sales" | "expense" | "inventory", data: any) => {
  if (io) {
    io.emit("dashboard-update", {
      type,
      data,
      timestamp: new Date().toISOString(),
    })
  }
}

export const broadcastNotification = (
  type: "success" | "error" | "info",
  title: string,
  message: string,
  targetRoom?: string,
) => {
  if (io) {
    const notification = {
      type,
      title,
      message,
      timestamp: new Date().toISOString(),
    }

    if (targetRoom) {
      io.to(targetRoom).emit("notification", notification)
    } else {
      io.emit("notification", notification)
    }
  }
}

export const broadcastUserActivity = (userId: string, action: string) => {
  if (io) {
    io.emit("user-activity", {
      userId,
      action,
      timestamp: new Date().toISOString(),
    })
  }
}
