import { broadcastDashboardUpdate, broadcastNotification, broadcastUserActivity } from "./server"

export class WebSocketEvents {
  static async onSalesCreate(salesData: any) {
    try {
      broadcastDashboardUpdate("sales", salesData)
      broadcastNotification(
        "success",
        "New Sale Recorded",
        `Sale of $${salesData.totalAmount} has been recorded`,
        `role:admin`,
      )
      broadcastUserActivity(salesData.createdBy, "created_sale")
    } catch (error) {
      console.error("Error broadcasting sales creation:", error)
    }
  }

  static async onExpenseCreate(expenseData: any) {
    try {
      broadcastDashboardUpdate("expense", expenseData)
      broadcastNotification(
        "info",
        "New Expense Added",
        `Expense of $${expenseData.amount} has been recorded`,
        `role:admin`,
      )
      broadcastUserActivity(expenseData.createdBy, "created_expense")
    } catch (error) {
      console.error("Error broadcasting expense creation:", error)
    }
  }

  static async onInventoryUpdate(inventoryData: any) {
    try {
      broadcastDashboardUpdate("inventory", inventoryData)
      broadcastNotification(
        "info",
        "Inventory Updated",
        `${inventoryData.itemName} inventory has been updated`,
        `role:manager`,
      )
      broadcastUserActivity(inventoryData.updatedBy, "updated_inventory")
    } catch (error) {
      console.error("Error broadcasting inventory update:", error)
    }
  }

  static async onUserLogin(userData: any) {
    try {
      broadcastUserActivity(userData._id, "logged_in")
      broadcastNotification("info", "User Activity", `${userData.name} has logged in`, `role:admin`)
    } catch (error) {
      console.error("Error broadcasting user login:", error)
    }
  }

  static async onUserLogout(userData: any) {
    try {
      broadcastUserActivity(userData._id, "logged_out")
    } catch (error) {
      console.error("Error broadcasting user logout:", error)
    }
  }

  static async onDataSubmission(type: "sales" | "expense" | "inventory", data: any, userId: string) {
    try {
      switch (type) {
        case "sales":
          await this.onSalesCreate({ ...data, createdBy: userId })
          break
        case "expense":
          await this.onExpenseCreate({ ...data, createdBy: userId })
          break
        case "inventory":
          await this.onInventoryUpdate({ ...data, updatedBy: userId })
          break
      }
    } catch (error) {
      console.error("Error handling data submission:", error)
    }
  }

  static async onError(error: string, userId?: string) {
    try {
      const targetRoom = userId ? `user:${userId}` : undefined
      broadcastNotification("error", "Error Occurred", error, targetRoom)
    } catch (err) {
      console.error("Error broadcasting error notification:", err)
    }
  }
}
