import { MongoClient, type Db } from "mongodb"

const uri = process.env.MONGODB_URI

if (!uri && process.env.NODE_ENV !== "development" && process.env.VERCEL_ENV !== "preview") {
  console.warn("MONGODB_URI environment variable is not set. Database features will be disabled.")
}

const options = {}

let client: MongoClient | null = null
let clientPromise: Promise<MongoClient> | null = null

if (uri) {
  if (process.env.NODE_ENV === "development") {
    // In development mode, use a global variable so that the value
    // is preserved across module reloads caused by HMR (Hot Module Replacement).
    const globalWithMongo = global as typeof globalThis & {
      _mongoClientPromise?: Promise<MongoClient>
    }

    if (!globalWithMongo._mongoClientPromise) {
      client = new MongoClient(uri, options)
      globalWithMongo._mongoClientPromise = client.connect()
    }
    clientPromise = globalWithMongo._mongoClientPromise
  } else {
    // In production mode, it's best to not use a global variable.
    client = new MongoClient(uri, options)
    clientPromise = client.connect()
  }
}

// Export a module-scoped MongoClient promise. By doing this in a
// separate module, the client can be shared across functions.
export default clientPromise

export async function getDatabase(): Promise<Db> {
  if (!clientPromise) {
    throw new Error("MongoDB connection not initialized. Please set MONGODB_URI environment variable.")
  }

  const client = await clientPromise
  return client.db("resturent")
}
