import { cookies } from "next/headers"
import type { NextRequest } from "next/server"

export interface SessionPayload {
  userId: string
  email: string
  role: string
  expiresAt: Date
}

// Simple base64 encoding for demo purposes (not secure for production)
function simpleEncode(payload: SessionPayload): string {
  const data = JSON.stringify({
    ...payload,
    expiresAt: payload.expiresAt.toISOString(),
  })
  return Buffer.from(data).toString("base64")
}

function simpleDecode(token: string): SessionPayload | null {
  try {
    const data = Buffer.from(token, "base64").toString("utf-8")
    const parsed = JSON.parse(data)
    return {
      ...parsed,
      expiresAt: new Date(parsed.expiresAt),
    }
  } catch {
    return null
  }
}

export async function encrypt(payload: SessionPayload) {
  return simpleEncode(payload)
}

export async function decrypt(input: string): Promise<SessionPayload> {
  const payload = simpleDecode(input)
  if (!payload) {
    throw new Error("Invalid session")
  }

  // Check if session is expired
  if (payload.expiresAt < new Date()) {
    throw new Error("Session expired")
  }

  return payload
}

export async function createSession(userId: string, email: string, role: string) {
  const expiresAt = new Date(Date.now() + 24 * 60 * 60 * 1000) // 24 hours
  const session = await encrypt({ userId, email, role, expiresAt })

  const cookieStore = await cookies()
  cookieStore.set("session", session, {
    expires: expiresAt,
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "lax",
    path: "/",
  })
}

export async function verifySession() {
  const cookieStore = await cookies()
  const cookie = cookieStore.get("session")?.value

  if (!cookie) return null

  try {
    const session = await decrypt(cookie)
    return session
  } catch (error) {
    return null
  }
}

export async function deleteSession() {
  const cookieStore = await cookies()
  cookieStore.delete("session")
}

export async function getSessionFromRequest(request: NextRequest) {
  const cookie = request.cookies.get("session")?.value

  if (!cookie) return null

  try {
    const session = await decrypt(cookie)
    return session
  } catch (error) {
    return null
  }
}
