"use client"

import { useEffect, useRef, useState } from "react"
import { io, type Socket } from "socket.io-client"
import { useToast } from "@/hooks/use-toast"
import type { ServerToClientEvents, ClientToServerEvents } from "@/lib/websocket/types"

type SocketType = Socket<ServerToClientEvents, ClientToServerEvents>

export function useWebSocket() {
  const [isConnected, setIsConnected] = useState(false)
  const [lastUpdate, setLastUpdate] = useState<any>(null)
  const socketRef = useRef<SocketType | null>(null)
  const { toast } = useToast()

  useEffect(() => {
    // Initialize socket connection
    const socket: SocketType = io({
      auth: {
        token: document.cookie
          .split("; ")
          .find((row) => row.startsWith("session="))
          ?.split("=")[1],
      },
    })

    socketRef.current = socket

    socket.on("connect", () => {
      setIsConnected(true)
      console.log("Connected to WebSocket server")
    })

    socket.on("disconnect", () => {
      setIsConnected(false)
      console.log("Disconnected from WebSocket server")
    })

    socket.on("dashboard-update", (data) => {
      setLastUpdate(data)

      // Show toast notification for updates
      toast({
        title: "Data Updated",
        description: `New ${data.type} data has been added`,
        duration: 3000,
      })
    })

    socket.on("notification", (data) => {
      toast({
        title: data.title,
        description: data.message,
        variant: data.type === "error" ? "destructive" : "default",
        duration: 5000,
      })
    })

    socket.on("user-activity", (data) => {
      console.log("User activity:", data)
    })

    return () => {
      socket.disconnect()
    }
  }, [toast])

  const joinRoom = (room: string) => {
    if (socketRef.current) {
      socketRef.current.emit("join-room", room)
    }
  }

  const leaveRoom = (room: string) => {
    if (socketRef.current) {
      socketRef.current.emit("leave-room", room)
    }
  }

  const ping = () => {
    if (socketRef.current) {
      socketRef.current.emit("ping")
    }
  }

  return {
    isConnected,
    lastUpdate,
    joinRoom,
    leaveRoom,
    ping,
    socket: socketRef.current,
  }
}
