"use client"

import { useState, useEffect } from "react"
import useSWR from "swr"

interface ReportsFilters {
  branchId?: string
  fromDate?: string
  toDate?: string
  category?: string
}

interface SalesReportData {
  _id: string
  date: string
  branchName: string
  cashInShop: number
  cashDelivery: number
  cashWolt: number
  cashBolt: number
  cashPyszne: number
  cashUberEats: number
  cashGlovo: number
  cardInShop: number
  cardDelivery: number
  cardWolt: number
  cardBolt: number
  cardPyszne: number
  cardUberEats: number
  cardGlovo: number
  totalCash: number
  totalCard: number
  totalSales: number
  notes?: string
}

interface ExpenseReportData {
  _id: string
  date: string
  branchName: string
  category: string
  subcategory: string
  amount: number
  description: string
  paymentMethod: string
  fakturaNo?: string
  notes?: string
  paymentDate?: string
}

interface InventoryReportData {
  _id: string
  date: string
  branchName: string
  type: string
  kurczak: number
  wolowina: number
  kraftowe: number
  majonez: number
  ketchup: number
  smabal: number
  yogurt: number
  pita_20_cm: number
  pita_30_cm: number
  gas: number
  notes?: string
}

interface IncomeStatementData {
  sales: {
    cashInShop: number
    cashDelivery: number
    cardInShop: number
    cardDelivery: number
    totalSales: number
  }
  expenses: {
    [category: string]: number
  }
  totalExpenses: number
  profitLoss: number
}

const fetcher = (url: string) => fetch(url).then((res) => res.json())

export function useSalesReport(filters: ReportsFilters = {}) {
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  // Build query parameters
  const queryParams = new URLSearchParams()
  if (filters.branchId && filters.branchId !== "all") {
    queryParams.append("branchId", filters.branchId)
  }
  if (filters.fromDate) {
    queryParams.append("fromDate", filters.fromDate)
  }
  if (filters.toDate) {
    queryParams.append("toDate", filters.toDate)
  }

  const queryString = queryParams.toString()
  const url = `/api/sales${queryString ? `?${queryString}` : ""}`

  const {
    data,
    error: swrError,
    mutate,
  } = useSWR<{
    success: boolean
    data: SalesReportData[]
    pagination: any
  }>(url, fetcher, {
    refreshInterval: 60000, // Refresh every minute
    revalidateOnFocus: true,
  })

  useEffect(() => {
    if (data || swrError) {
      setIsLoading(false)
    }
    if (swrError) {
      setError("Failed to load sales report data")
    } else {
      setError(null)
    }
  }, [data, swrError])

  return {
    data: data?.data || [],
    isLoading,
    error,
    mutate,
  }
}

export function useExpenseReport(filters: ReportsFilters = {}) {
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  // Build query parameters
  const queryParams = new URLSearchParams()
  if (filters.branchId && filters.branchId !== "all") {
    queryParams.append("branchId", filters.branchId)
  }
  if (filters.fromDate) {
    queryParams.append("fromDate", filters.fromDate)
  }
  if (filters.toDate) {
    queryParams.append("toDate", filters.toDate)
  }
  if (filters.category) {
    queryParams.append("category", filters.category)
  }

  const queryString = queryParams.toString()
  const url = `/api/expenses${queryString ? `?${queryString}` : ""}`

  const {
    data,
    error: swrError,
    mutate,
  } = useSWR<{
    success: boolean
    data: ExpenseReportData[]
    pagination: any
  }>(url, fetcher, {
    refreshInterval: 60000, // Refresh every minute
    revalidateOnFocus: true,
  })

  useEffect(() => {
    if (data || swrError) {
      setIsLoading(false)
    }
    if (swrError) {
      setError("Failed to load expense report data")
    } else {
      setError(null)
    }
  }, [data, swrError])

  return {
    data: data?.data || [],
    isLoading,
    error,
    mutate,
  }
}

export function useInventoryReport(filters: ReportsFilters = {}) {
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  // Build query parameters
  const queryParams = new URLSearchParams()
  if (filters.branchId && filters.branchId !== "all") {
    queryParams.append("branchId", filters.branchId)
  }
  if (filters.fromDate) {
    queryParams.append("fromDate", filters.fromDate)
  }
  if (filters.toDate) {
    queryParams.append("toDate", filters.toDate)
  }

  const queryString = queryParams.toString()
  const url = `/api/inventory${queryString ? `?${queryString}` : ""}`

  const {
    data,
    error: swrError,
    mutate,
  } = useSWR<{
    success: boolean
    data: InventoryReportData[]
    pagination: any
  }>(url, fetcher, {
    refreshInterval: 60000, // Refresh every minute
    revalidateOnFocus: true,
  })

  useEffect(() => {
    if (data || swrError) {
      setIsLoading(false)
    }
    if (swrError) {
      setError("Failed to load inventory report data")
    } else {
      setError(null)
    }
  }, [data, swrError])

  return {
    data: data?.data || [],
    isLoading,
    error,
    mutate,
  }
}

export function useIncomeStatement(filters: ReportsFilters = {}) {
  const { data: salesData, isLoading: salesLoading, error: salesError } = useSalesReport(filters)
  const { data: expenseData, isLoading: expenseLoading, error: expenseError } = useExpenseReport(filters)

  const [incomeStatement, setIncomeStatement] = useState<IncomeStatementData>({
    sales: {
      cashInShop: 0,
      cashDelivery: 0,
      cardInShop: 0,
      cardDelivery: 0,
      totalSales: 0,
    },
    expenses: {},
    totalExpenses: 0,
    profitLoss: 0,
  })

  useEffect(() => {
    if (salesData && expenseData) {
      // Calculate sales totals
      const salesTotals = salesData.reduce(
        (acc, sale) => ({
          cashInShop: acc.cashInShop + sale.cashInShop,
          cashDelivery: acc.cashDelivery + sale.cashDelivery,
          cardInShop: acc.cardInShop + sale.cardInShop,
          cardDelivery: acc.cardDelivery + sale.cardDelivery,
          totalSales: acc.totalSales + sale.totalSales,
        }),
        {
          cashInShop: 0,
          cashDelivery: 0,
          cardInShop: 0,
          cardDelivery: 0,
          totalSales: 0,
        },
      )

      // Calculate expenses by category
      const expensesByCategory = expenseData.reduce(
        (acc, expense) => {
          const category = expense.category || "Other"
          acc[category] = (acc[category] || 0) + expense.amount
          return acc
        },
        {} as { [key: string]: number },
      )

      const totalExpenses = Object.values(expensesByCategory).reduce((sum, amount) => sum + amount, 0)
      const profitLoss = salesTotals.totalSales - totalExpenses

      setIncomeStatement({
        sales: salesTotals,
        expenses: expensesByCategory,
        totalExpenses,
        profitLoss,
      })
    }
  }, [salesData, expenseData])

  return {
    data: incomeStatement,
    isLoading: salesLoading || expenseLoading,
    error: salesError || expenseError,
  }
}

export function useSalaryReport(filters: ReportsFilters = {}) {
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  // Build query parameters for salary-related expenses
  const queryParams = new URLSearchParams()
  if (filters.branchId && filters.branchId !== "all") {
    queryParams.append("branchId", filters.branchId)
  }
  if (filters.fromDate) {
    queryParams.append("fromDate", filters.fromDate)
  }
  if (filters.toDate) {
    queryParams.append("toDate", filters.toDate)
  }
  queryParams.append("category", "salaries") // Filter for salary expenses

  const queryString = queryParams.toString()
  const url = `/api/expenses${queryString ? `?${queryString}` : ""}`

  const {
    data,
    error: swrError,
    mutate,
  } = useSWR<{
    success: boolean
    data: ExpenseReportData[]
    pagination: any
  }>(url, fetcher, {
    refreshInterval: 60000,
    revalidateOnFocus: true,
  })

  useEffect(() => {
    if (data || swrError) {
      setIsLoading(false)
    }
    if (swrError) {
      setError("Failed to load salary report data")
    } else {
      setError(null)
    }
  }, [data, swrError])

  return {
    data: data?.data || [],
    isLoading,
    error,
    mutate,
  }
}
