"use client"

import { useState } from "react"
import { toast } from "@/hooks/use-toast"

interface SubmissionState {
  isLoading: boolean
  error: string | null
}

export function useFormSubmission() {
  const [state, setState] = useState<SubmissionState>({
    isLoading: false,
    error: null,
  })

  const submitSalesData = async (salesData: any) => {
    setState({ isLoading: true, error: null })
    console.log("[v0] Submitting sales data:", salesData) // Debug log

    try {
      const response = await fetch("/api/sales", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(salesData),
      })

      const result = await response.json()
      console.log("[v0] Sales API response:", result) // Debug log

      if (!result.success) {
        throw new Error(result.error || "Failed to submit sales data")
      }

      toast({
        title: "Success",
        description: "Sales data has been saved successfully",
      })

      setState({ isLoading: false, error: null })
      return result.data
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : "An error occurred"
      console.error("[v0] Sales submission error:", error) // Debug log
      setState({ isLoading: false, error: errorMessage })

      toast({
        title: "Error",
        description: errorMessage,
        variant: "destructive",
      })

      throw error
    }
  }

  const submitExpenseData = async (expenseData: any) => {
    setState({ isLoading: true, error: null })
    console.log("[v0] Submitting expense data:", expenseData) // Debug log

    try {
      const response = await fetch("/api/expenses", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(expenseData),
      })

      const result = await response.json()
      console.log("[v0] Expense API response:", result) // Debug log

      if (!result.success) {
        throw new Error(result.error || "Failed to submit expense data")
      }

      toast({
        title: "Success",
        description: "Expense data has been saved successfully",
      })

      setState({ isLoading: false, error: null })
      return result.data
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : "An error occurred"
      console.error("[v0] Expense submission error:", error) // Debug log
      setState({ isLoading: false, error: errorMessage })

      toast({
        title: "Error",
        description: errorMessage,
        variant: "destructive",
      })

      throw error
    }
  }

  const submitInventoryData = async (inventoryData: any) => {
    setState({ isLoading: true, error: null })
    console.log("[v0] Submitting inventory data:", inventoryData) // Debug log

    try {
      const response = await fetch("/api/inventory", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(inventoryData),
      })

      const result = await response.json()
      console.log("[v0] Inventory API response:", result) // Debug log

      if (!result.success) {
        throw new Error(result.error || "Failed to submit inventory data")
      }

      toast({
        title: "Success",
        description: "Inventory data has been saved successfully",
      })

      setState({ isLoading: false, error: null })
      return result.data
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : "An error occurred"
      console.error("[v0] Inventory submission error:", error) // Debug log
      setState({ isLoading: false, error: errorMessage })

      toast({
        title: "Error",
        description: errorMessage,
        variant: "destructive",
      })

      throw error
    }
  }

  return {
    ...state,
    submitSalesData,
    submitExpenseData,
    submitInventoryData,
  }
}

// Helper function to transform form data to API format
export function transformSalesFormData(formData: any, branchId: string, date: string) {
  return {
    date,
    branchId,
    cashInShop: Number.parseFloat(formData.cash_in_shop) || 0,
    cashDelivery: Number.parseFloat(formData.cash_delivery) || 0,
    cashWolt: Number.parseFloat(formData.cash_wolt) || 0,
    cashBolt: Number.parseFloat(formData.cash_bolt) || 0,
    cashPyszne: Number.parseFloat(formData.cash_pyszne_pi) || 0,
    cashUberEats: Number.parseFloat(formData.cash_uber_eats) || 0,
    cashGlovo: Number.parseFloat(formData.cash_glovo) || 0,
    cardInShop: Number.parseFloat(formData.card_in_shop) || 0,
    cardDelivery: Number.parseFloat(formData.card_delivery) || 0,
    cardWolt: Number.parseFloat(formData.card_wolt) || 0,
    cardBolt: Number.parseFloat(formData.card_bolt) || 0,
    cardPyszne: Number.parseFloat(formData.card_pyszne_pi) || 0,
    cardUberEats: Number.parseFloat(formData.card_uber_eats) || 0,
    cardGlovo: Number.parseFloat(formData.card_glovo) || 0,
    notes: formData.notes || "",
  }
}

export function transformExpenseFormData(
  formData: any,
  branchId: string,
  date: string,
  category: string,
  subcategory: string,
) {
  return {
    date,
    branchId,
    category,
    subcategory,
    amount: Number.parseFloat(formData.brutto) || 0,
    description: formData.itemName || "",
    paymentMethod: formData.paymentMethod || "cash",
    employeeId: formData.employee || null,
    hourlyRate: Number.parseFloat(formData.hourlyRate) || null,
    totalHours: Number.parseFloat(formData.totalHours) || null,
    totalSalary: Number.parseFloat(formData.totalSalary) || null,
    received: Number.parseFloat(formData.received) || null,
    payable: Number.parseFloat(formData.payable) || null,
    netto: Number.parseFloat(formData.netto) || null,
    vat: Number.parseFloat(formData.vat) || null,
    brutto: Number.parseFloat(formData.brutto) || null,
    fakturaNo: formData.fakturaNo || "",
    notes: formData.notes || "",
    paymentDate: formData.paymentDate || null,
  }
}

export function transformInventoryFormData(formData: any, branchId: string, date: string, type: string) {
  // Map form types to database types
  const dbType = type === "sub" ? "daily-consumption" : "stock-level"

  return {
    date,
    branchId,
    type: dbType, // Use proper database type
    kurczak: Number.parseFloat(formData.kurczak) || 0,
    wolowina: Number.parseFloat(formData.wolowina) || 0,
    kraftowe: Number.parseFloat(formData.kraftowe) || 0,
    majonez: Number.parseFloat(formData.majonez) || 0,
    ketchup: Number.parseFloat(formData.ketchup) || 0,
    smabal: Number.parseFloat(formData.smabal) || 0,
    yogurt: Number.parseFloat(formData.yogurt) || 0,
    pita20cm: Number.parseFloat(formData.pita_20_cm) || 0,
    pita30cm: Number.parseFloat(formData.pita_30_cm) || 0,
    gas: Number.parseFloat(formData.gas) || 0,
    notes: formData.notes || "",
  }
}
