"use client"

import { useState, useEffect } from "react"
import useSWR from "swr"

interface DashboardMetrics {
  totalSales: { value: number; change: number; trend: "up" | "down" | "neutral" }
  totalExpenses: { value: number; change: number; trend: "up" | "down" | "neutral" }
  netProfit: { value: number; change: number; trend: "up" | "down" | "neutral" }
  activeBranches: { value: number; change: number; trend: "up" | "down" | "neutral" }
}

interface MonthlyData {
  month: string
  sales: number
  expenses: number
}

interface DashboardData {
  metrics: DashboardMetrics
  monthlyData: MonthlyData[]
}

interface DashboardFilters {
  branchId?: string
  fromDate?: string
  toDate?: string
}

const fetcher = (url: string) => fetch(url).then((res) => res.json())

export function useDashboardData(filters: DashboardFilters = {}) {
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  // Build query parameters
  const queryParams = new URLSearchParams()
  if (filters.branchId && filters.branchId !== "all") {
    queryParams.append("branchId", filters.branchId)
  }
  if (filters.fromDate) {
    queryParams.append("fromDate", filters.fromDate)
  }
  if (filters.toDate) {
    queryParams.append("toDate", filters.toDate)
  }

  const queryString = queryParams.toString()
  const url = `/api/dashboard/metrics${queryString ? `?${queryString}` : ""}`

  const {
    data,
    error: swrError,
    mutate,
  } = useSWR<{ success: boolean; data: DashboardData }>(url, fetcher, {
    refreshInterval: 30000, // Refresh every 30 seconds for real-time updates
    revalidateOnFocus: true,
    revalidateOnReconnect: true,
  })

  useEffect(() => {
    if (data || swrError) {
      setIsLoading(false)
    }
    if (swrError) {
      setError("Failed to load dashboard data")
    } else {
      setError(null)
    }
  }, [data, swrError])

  return {
    data: data?.data,
    isLoading,
    error,
    mutate, // For manual refresh
  }
}

export function useBranches() {
  const { data, error, mutate } = useSWR<{
    success: boolean
    data: Array<{ _id: string; name: string; isActive: boolean }>
  }>("/api/branches?active=true", fetcher)

  return {
    branches: data?.data || [],
    isLoading: !error && !data,
    error,
    mutate,
  }
}

export function useEmployees(branchId?: string) {
  const queryParams = new URLSearchParams()
  if (branchId && branchId !== "all") {
    queryParams.append("branchId", branchId)
  }

  const queryString = queryParams.toString()
  const url = `/api/employees${queryString ? `?${queryString}` : ""}`

  const { data, error, mutate } = useSWR<{
    success: boolean
    data: Array<{
      _id: string
      name: string
      position: string
      email: string
      branchId: string
      branchName: string
      hourlyRate: number
      status: "active" | "inactive"
    }>
  }>(branchId && branchId !== "" ? url : null, fetcher, {
    revalidateOnFocus: false,
    revalidateOnReconnect: true,
  })

  console.log("[v0] useEmployees hook:", { branchId, url, data: data?.data, error })

  return {
    employees: data?.data || [],
    isLoading: !error && !data && !!branchId && branchId !== "",
    error,
    mutate,
  }
}
