"use client"

import { useEffect, useState } from "react"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { useWebSocket } from "@/hooks/use-websocket"
import { TrendingUp, TrendingDown, Activity, Users, Wifi, WifiOff } from "lucide-react"

interface DashboardMetrics {
  totalSales: number
  totalExpenses: number
  activeUsers: number
  lastUpdate: string
}

export default function RealTimeDashboard() {
  const { isConnected, lastUpdate, ping } = useWebSocket()
  const [metrics, setMetrics] = useState<DashboardMetrics>({
    totalSales: 0,
    totalExpenses: 0,
    activeUsers: 0,
    lastUpdate: new Date().toISOString(),
  })

  useEffect(() => {
    if (lastUpdate) {
      // Update metrics based on real-time data
      setMetrics((prev) => ({
        ...prev,
        lastUpdate: lastUpdate.timestamp,
      }))

      // Update specific metrics based on update type
      if (lastUpdate.type === "sales") {
        setMetrics((prev) => ({
          ...prev,
          totalSales: prev.totalSales + (lastUpdate.data.totalAmount || 0),
        }))
      } else if (lastUpdate.type === "expense") {
        setMetrics((prev) => ({
          ...prev,
          totalExpenses: prev.totalExpenses + (lastUpdate.data.amount || 0),
        }))
      }
    }
  }, [lastUpdate])

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount)
  }

  const formatTime = (timestamp: string) => {
    return new Date(timestamp).toLocaleTimeString()
  }

  return (
    <div className="space-y-6">
      {/* Connection Status */}
      <Card>
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <CardTitle className="text-lg">Real-time Status</CardTitle>
            <div className="flex items-center gap-2">
              {isConnected ? (
                <>
                  <Wifi className="h-4 w-4 text-green-500" />
                  <Badge variant="secondary" className="bg-green-100 text-green-700">
                    Connected
                  </Badge>
                </>
              ) : (
                <>
                  <WifiOff className="h-4 w-4 text-red-500" />
                  <Badge variant="destructive">Disconnected</Badge>
                </>
              )}
              <Button variant="outline" size="sm" onClick={ping}>
                Test Connection
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <p className="text-sm text-muted-foreground">Last update: {formatTime(metrics.lastUpdate)}</p>
        </CardContent>
      </Card>

      {/* Real-time Metrics */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Sales</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{formatCurrency(metrics.totalSales)}</div>
            <p className="text-xs text-muted-foreground">Real-time updates</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Expenses</CardTitle>
            <TrendingDown className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{formatCurrency(metrics.totalExpenses)}</div>
            <p className="text-xs text-muted-foreground">Real-time updates</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Active Users</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{metrics.activeUsers}</div>
            <p className="text-xs text-muted-foreground">Currently online</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">System Status</CardTitle>
            <Activity className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{isConnected ? "Online" : "Offline"}</div>
            <p className="text-xs text-muted-foreground">WebSocket connection</p>
          </CardContent>
        </Card>
      </div>

      {/* Recent Activity */}
      <Card>
        <CardHeader>
          <CardTitle>Recent Activity</CardTitle>
          <CardDescription>Live updates from your business operations</CardDescription>
        </CardHeader>
        <CardContent>
          {lastUpdate ? (
            <div className="space-y-2">
              <div className="flex items-center justify-between p-3 bg-muted rounded-lg">
                <div>
                  <p className="font-medium">New {lastUpdate.type} record added</p>
                  <p className="text-sm text-muted-foreground">{formatTime(lastUpdate.timestamp)}</p>
                </div>
                <Badge variant="outline">{lastUpdate.type}</Badge>
              </div>
            </div>
          ) : (
            <p className="text-muted-foreground">No recent activity. Data will appear here in real-time.</p>
          )}
        </CardContent>
      </Card>
    </div>
  )
}
