"use client"

import type React from "react"
import { useState } from "react"
import { useRouter } from "next/navigation"
import Link from "next/link"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import {
  Menu,
  Settings,
  User,
  Database,
  Home,
  Eye,
  BarChart3,
  TrendingUp,
  DollarSign,
  Users,
  MapPin,
  Shield,
  Crown,
  UsersIcon,
  Download,
} from "lucide-react"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { useBranches } from "@/hooks/use-dashboard-data"

interface DashboardLayoutProps {
  children: React.ReactNode
}

export default function DashboardLayout({ children }: DashboardLayoutProps) {
  const router = useRouter()
  const { branches, isLoading: branchesLoading } = useBranches()

  const [sidebarOpen, setSidebarOpen] = useState(false)
  const [addDataModalOpen, setAddDataModalOpen] = useState(false)
  const [addEmployeeModalOpen, setAddEmployeeModalOpen] = useState(false)
  const [profileViewModalOpen, setProfileViewModalOpen] = useState(false)
  const [changePasswordModalOpen, setChangePasswordModalOpen] = useState(false)
  const [exportReportModalOpen, setExportReportModalOpen] = useState(false)
  const [addDataForm, setAddDataForm] = useState({
    date: new Date().toISOString().split("T")[0],
    branch: "",
  })

  const [exportReportForm, setExportReportForm] = useState({
    fromDate: new Date().toISOString().split("T")[0],
    toDate: new Date().toISOString().split("T")[0],
    branch: "",
  })

  const [addEmployeeForm, setAddEmployeeForm] = useState({
    branch_id: "",
    name: "",
    email: "",
    phone: "",
    position: "",
    salary: "",
  })

  const handleAddDataSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    console.log("[v0] Add Data Form submitted:", addDataForm)
    setAddDataModalOpen(false)
    const selectedBranch = branches.find((b) => b._id === addDataForm.branch)
    const params = new URLSearchParams({
      date: addDataForm.date,
      branch: selectedBranch?.name || addDataForm.branch,
      branchId: addDataForm.branch,
    })
    console.log("[v0] Navigating to add-data with params:", params.toString())
    router.push(`/add-data?${params.toString()}`)
    setAddDataForm({
      date: new Date().toISOString().split("T")[0],
      branch: "",
    })
  }

  const handleNavigation = (path: string) => {
    console.log("[v0] Navigating to:", path)
    router.push(path)
  }

  const handleLogout = async () => {
    console.log("[v0] Logout clicked")
    // For now, just redirect to login or home
    router.push("/")
  }

  const handleAddEmployeeSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    console.log("Add Employee Form:", addEmployeeForm)
    setAddEmployeeModalOpen(false)
    setAddEmployeeForm({
      branch_id: "",
      name: "",
      email: "",
      phone: "",
      position: "",
      salary: "",
    })
  }

  const handleExportReportSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    console.log("[v0] Export Report Form submitted:", exportReportForm)
    setExportReportModalOpen(false)
    const selectedBranch = branches.find((b) => b._id === exportReportForm.branch)
    const params = new URLSearchParams({
      fromDate: exportReportForm.fromDate,
      toDate: exportReportForm.toDate,
      branch: selectedBranch?.name || exportReportForm.branch,
      branchId: exportReportForm.branch,
    })
    console.log("[v0] Navigating to export-reports with params:", params.toString())
    router.push(`/export-reports?${params.toString()}`)
    setExportReportForm({
      fromDate: new Date().toISOString().split("T")[0],
      toDate: new Date().toISOString().split("T")[0],
      branch: "",
    })

    setTimeout(() => {
      alert("Export Report generation started! You will be redirected to the export page.")
    }, 100)
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50/30">
      {/* Header */}
      <header className="sticky top-0 z-40 w-full border-b bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
        <div className="container flex h-12 sm:h-14 items-center justify-between px-3 sm:px-4">
          <div className="flex items-center space-x-2 sm:space-x-4">
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setSidebarOpen(!sidebarOpen)}
              className="md:hidden h-8 w-8 sm:h-10 sm:w-10"
            >
              <Menu className="h-4 w-4 sm:h-5 sm:w-5" />
            </Button>

            <div className="flex items-center space-x-2">
              <div className="h-6 w-6 sm:h-8 sm:w-8 rounded-lg bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center">
                <span className="text-white font-bold text-xs sm:text-sm">D</span>
              </div>
              <h1 className="text-lg sm:text-xl font-bold hidden xs:block">Dashboard</h1>
            </div>
          </div>

          {/* Profile Dropdown */}
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" className="relative h-8 w-8 sm:h-10 sm:w-10 rounded-full">
                <Avatar className="h-8 w-8 sm:h-10 sm:w-10">
                  <AvatarImage src="/placeholder.svg?height=40&width=40" alt="Profile" />
                  <AvatarFallback className="bg-gradient-to-br from-blue-500 to-purple-600 text-white">
                    <User className="h-3 w-3 sm:h-5 sm:w-5" />
                  </AvatarFallback>
                </Avatar>
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent
              className="w-48 sm:w-56"
              align="end"
              side="bottom"
              alignOffset={0}
              sideOffset={8}
              forceMount
            >
              <DropdownMenuLabel className="font-normal">
                <div className="flex flex-col space-y-1">
                  <p className="text-sm font-medium leading-none">Admin User</p>
                  <p className="text-xs leading-none text-muted-foreground">admin@example.com</p>
                  <p className="text-xs leading-none text-muted-foreground capitalize">admin</p>
                </div>
              </DropdownMenuLabel>
              <DropdownMenuSeparator />
              <DropdownMenuItem
                onClick={() => {
                  setProfileViewModalOpen(true)
                  setTimeout(() => {
                    console.log("[v0] Profile view opened")
                  }, 100)
                }}
              >
                <Eye className="mr-2 h-4 w-4 sm:h-5 sm:w-5" />
                <span>View Profile</span>
              </DropdownMenuItem>
              <DropdownMenuItem
                onClick={() => {
                  setChangePasswordModalOpen(true)
                  setTimeout(() => {
                    console.log("[v0] Change password modal opened")
                  }, 100)
                }}
              >
                <Settings className="mr-2 h-4 w-4" />
                <span>Change Password</span>
              </DropdownMenuItem>
              <DropdownMenuSeparator />
              <DropdownMenuItem onClick={handleLogout}>
                <span>Logout</span>
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </header>

      <div className="flex">
        {sidebarOpen && (
          <div className="fixed inset-0 bg-black/50 z-40 md:hidden" onClick={() => setSidebarOpen(false)} />
        )}

        <aside
          className={`fixed inset-y-0 left-0 z-50 w-64 sm:w-72 bg-card border-r transform transition-transform duration-200 ease-in-out md:relative md:translate-x-0 ${sidebarOpen ? "translate-x-0" : "-translate-x-full"}`}
        >
          <div className="flex h-full flex-col">
            <div className="flex-1 py-6 overflow-y-auto">
              <nav className="space-y-6 px-4">
                {/* Dashboard Section */}
                <div>
                  <Link href="/" className="w-full">
                    <Button variant="ghost" className="w-full justify-start text-sm bg-accent h-10 font-medium">
                      <Home className="mr-3 h-5 w-5" />
                      Dashboard
                    </Button>
                  </Link>
                </div>

                {/* Business Operations Section */}
                <div>
                  <h3 className="mb-3 px-3 text-xs font-semibold text-muted-foreground uppercase tracking-wider">
                    Business Operations
                  </h3>
                  <div className="space-y-2">
                    {/* Add Data Button with Modal */}
                    <Dialog open={addDataModalOpen} onOpenChange={setAddDataModalOpen}>
                      <DialogTrigger asChild>
                        <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                          <Database className="mr-3 h-5 w-5" />
                          Add Data
                        </Button>
                      </DialogTrigger>
                      <DialogContent className="max-w-[95vw] sm:max-w-md p-4 sm:p-6">
                        <DialogHeader>
                          <DialogTitle>Add Data</DialogTitle>
                          <DialogDescription>Select a date and branch to add new data entries.</DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleAddDataSubmit} className="space-y-4">
                          <div className="space-y-2">
                            <Label htmlFor="date">
                              Date <span className="text-red-500">*</span>
                            </Label>
                            <Input
                              id="date"
                              type="date"
                              value={addDataForm.date}
                              onChange={(e) => setAddDataForm({ ...addDataForm, date: e.target.value })}
                              required
                            />
                            <div className="text-xs text-gray-500">Selected: {addDataForm.date}</div>
                          </div>
                          <div className="space-y-2">
                            <Label htmlFor="branch">
                              Branch <span className="text-red-500">*</span>
                            </Label>
                            <Select
                              value={addDataForm.branch}
                              onValueChange={(value) => setAddDataForm({ ...addDataForm, branch: value })}
                              required
                            >
                              <SelectTrigger>
                                <SelectValue placeholder="Select Branch" />
                              </SelectTrigger>
                              <SelectContent>
                                {branchesLoading ? (
                                  <SelectItem value="" disabled>
                                    Loading branches...
                                  </SelectItem>
                                ) : branches.length > 0 ? (
                                  branches.map((branch) => (
                                    <SelectItem key={branch._id} value={branch._id}>
                                      {branch.name}
                                    </SelectItem>
                                  ))
                                ) : (
                                  <SelectItem value="" disabled>
                                    No branches available
                                  </SelectItem>
                                )}
                              </SelectContent>
                            </Select>
                            <div className="text-xs text-gray-500">
                              Selected: {addDataForm.branch || "None"}
                              {addDataForm.branch && (
                                <span className="ml-2">
                                  ({branches.find((b) => b._id === addDataForm.branch)?.name || "Unknown"})
                                </span>
                              )}
                            </div>
                          </div>
                          <div className="flex justify-end space-x-2 pt-4">
                            <Button type="button" variant="outline" onClick={() => setAddDataModalOpen(false)}>
                              Cancel
                            </Button>
                            <Button type="submit" className="bg-blue-600 hover:bg-blue-700">
                              Continue
                            </Button>
                          </div>
                        </form>
                      </DialogContent>
                    </Dialog>
                  </div>
                </div>

                {/* Reports Section */}
                <div>
                  <h3 className="mb-3 px-3 text-xs font-semibold text-muted-foreground uppercase tracking-wider">
                    Reports & Analytics
                  </h3>
                  <div className="space-y-2">
                    {/* Export Report Button with Modal */}
                    <Dialog open={exportReportModalOpen} onOpenChange={setExportReportModalOpen}>
                      <DialogTrigger asChild>
                        <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                          <Download className="mr-3 h-5 w-5" />
                          Export Report
                        </Button>
                      </DialogTrigger>
                      <DialogContent className="max-w-[95vw] sm:max-w-md p-4 sm:p-6">
                        <DialogHeader>
                          <DialogTitle>Export Report</DialogTitle>
                          <DialogDescription>
                            Select date range and branch to export comprehensive reports.
                          </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleExportReportSubmit} className="space-y-4">
                          <div className="space-y-2">
                            <Label htmlFor="fromDate">
                              From Date <span className="text-red-500">*</span>
                            </Label>
                            <Input
                              id="fromDate"
                              type="date"
                              value={exportReportForm.fromDate}
                              onChange={(e) => setExportReportForm({ ...exportReportForm, fromDate: e.target.value })}
                              required
                            />
                          </div>
                          <div className="space-y-2">
                            <Label htmlFor="toDate">
                              To Date <span className="text-red-500">*</span>
                            </Label>
                            <Input
                              id="toDate"
                              type="date"
                              value={exportReportForm.toDate}
                              onChange={(e) => setExportReportForm({ ...exportReportForm, toDate: e.target.value })}
                              required
                            />
                          </div>
                          <div className="space-y-2">
                            <Label htmlFor="exportBranch">
                              Branch <span className="text-red-500">*</span>
                            </Label>
                            <Select
                              value={exportReportForm.branch}
                              onValueChange={(value) => setExportReportForm({ ...exportReportForm, branch: value })}
                              required
                            >
                              <SelectTrigger>
                                <SelectValue placeholder="Select Branch" />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectItem value="all">All Branches</SelectItem>
                                {branchesLoading ? (
                                  <SelectItem value="" disabled>
                                    Loading branches...
                                  </SelectItem>
                                ) : branches.length > 0 ? (
                                  branches.map((branch) => (
                                    <SelectItem key={branch._id} value={branch._id}>
                                      {branch.name}
                                    </SelectItem>
                                  ))
                                ) : (
                                  <SelectItem value="" disabled>
                                    No branches available
                                  </SelectItem>
                                )}
                              </SelectContent>
                            </Select>
                          </div>
                          <div className="flex justify-end space-x-2 pt-4">
                            <Button type="button" variant="outline" onClick={() => setExportReportModalOpen(false)}>
                              Cancel
                            </Button>
                            <Button type="submit" className="bg-green-600 hover:bg-green-700">
                              Generate Export
                            </Button>
                          </div>
                        </form>
                      </DialogContent>
                    </Dialog>

                    <Link href="/sales-report" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <TrendingUp className="mr-3 h-5 w-5" />
                        Sales Report
                      </Button>
                    </Link>

                    <Link href="/inventory-report" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <BarChart3 className="mr-3 h-5 w-5" />
                        Inventory Report
                      </Button>
                    </Link>

                    <Link href="/expense-report" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <DollarSign className="mr-3 h-5 w-5" />
                        Expense Report
                      </Button>
                    </Link>
                  </div>
                </div>

                {/* Branch Management Section */}
                <div>
                  <h3 className="mb-3 px-3 text-xs font-semibold text-muted-foreground uppercase tracking-wider">
                    Branch Management
                  </h3>
                  <div className="space-y-2">
                    <Link href="/branches" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <MapPin className="mr-3 h-5 w-5" />
                        All Branches
                      </Button>
                    </Link>

                    <Link href="/assign-branch" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <Users className="mr-3 h-5 w-5" />
                        Assign Branch
                      </Button>
                    </Link>
                  </div>
                </div>

                {/* User Management Section */}
                <div>
                  <h3 className="mb-3 px-3 text-xs font-semibold text-muted-foreground uppercase tracking-wider">
                    User Management
                  </h3>
                  <div className="space-y-2">
                    <Link href="/admin" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <Shield className="mr-3 h-5 w-5" />
                        Add Admin
                      </Button>
                    </Link>

                    <Link href="/manager" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <User className="mr-3 h-5 w-5" />
                        Add Manager
                      </Button>
                    </Link>

                    <Link href="/owner" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <Crown className="mr-3 h-5 w-5" />
                        Add Owner
                      </Button>
                    </Link>

                    <Link href="/employees" className="w-full">
                      <Button variant="ghost" className="w-full justify-start text-sm h-10 font-medium">
                        <UsersIcon className="mr-3 h-5 w-5" />
                        View Employees
                      </Button>
                    </Link>
                  </div>
                </div>
              </nav>
            </div>
          </div>
        </aside>

        {/* Main Content */}
        <main className="flex-1 min-w-0 overflow-auto bg-gradient-to-br from-slate-50 to-blue-50/30">{children}</main>
      </div>

      {/* Profile View Modal */}
      <Dialog open={profileViewModalOpen} onOpenChange={setProfileViewModalOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-md p-4 sm:p-6">
          <DialogHeader>
            <DialogTitle>Profile Information</DialogTitle>
            <DialogDescription>View your profile details</DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="flex items-center space-x-4">
              <Avatar className="h-16 w-16">
                <AvatarImage src="/placeholder.svg?height=64&width=64" alt="Profile" />
                <AvatarFallback className="bg-gradient-to-br from-blue-500 to-purple-600 text-white text-xl">
                  <User className="h-8 w-8" />
                </AvatarFallback>
              </Avatar>
              <div>
                <h3 className="text-lg font-semibold">Admin User</h3>
                <p className="text-sm text-muted-foreground">admin@example.com</p>
                <p className="text-sm text-muted-foreground capitalize">Administrator</p>
              </div>
            </div>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-sm font-medium">Role:</span>
                <span className="text-sm">Administrator</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm font-medium">Status:</span>
                <span className="text-sm text-green-600">Active</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm font-medium">Last Login:</span>
                <span className="text-sm">{new Date().toLocaleDateString()}</span>
              </div>
            </div>
            <div className="flex justify-end">
              <Button
                onClick={() => {
                  setProfileViewModalOpen(false)
                  setTimeout(() => {
                    alert("Profile information viewed successfully!")
                  }, 100)
                }}
              >
                Close
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Change Password Modal */}
      <Dialog open={changePasswordModalOpen} onOpenChange={setChangePasswordModalOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-md p-4 sm:p-6">
          <DialogHeader>
            <DialogTitle>Change Password</DialogTitle>
            <DialogDescription>Update your account password</DialogDescription>
          </DialogHeader>
          <form
            className="space-y-4"
            onSubmit={(e) => {
              e.preventDefault()
              setChangePasswordModalOpen(false)
              setTimeout(() => {
                alert("Password changed successfully! Please login again with your new password.")
              }, 100)
            }}
          >
            <div className="space-y-2">
              <Label htmlFor="currentPassword">Current Password</Label>
              <Input id="currentPassword" type="password" placeholder="Enter current password" required />
            </div>
            <div className="space-y-2">
              <Label htmlFor="newPassword">New Password</Label>
              <Input id="newPassword" type="password" placeholder="Enter new password" required />
            </div>
            <div className="space-y-2">
              <Label htmlFor="confirmPassword">Confirm New Password</Label>
              <Input id="confirmPassword" type="password" placeholder="Confirm new password" required />
            </div>
            <div className="flex justify-end space-x-2 pt-4">
              <Button type="button" variant="outline" onClick={() => setChangePasswordModalOpen(false)}>
                Cancel
              </Button>
              <Button type="submit" className="bg-blue-600 hover:bg-blue-700">
                Update Password
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  )
}
