"use client"

import type React from "react"
import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Filter, BarChart3, Edit, Eye, Trash2, Loader2 } from "lucide-react"
import DashboardLayout from "@/components/dashboard-layout"
import { useSalesReport } from "@/hooks/use-reports-data"
import { useToast } from "@/hooks/use-toast"

export default function SalesReportPage() {
  const [fromDate, setFromDate] = useState("")
  const [toDate, setToDate] = useState("")
  const [selectedBranch, setSelectedBranch] = useState("all")
  const [branches, setBranches] = useState<Array<{ _id: string; name: string }>>([])

  const [selectedSale, setSelectedSale] = useState<any>(null)
  const [isViewModalOpen, setIsViewModalOpen] = useState(false)
  const [isEditModalOpen, setIsEditModalOpen] = useState(false)
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false)
  const [editFormData, setEditFormData] = useState<any>({})
  const { toast } = useToast()

  const {
    data: salesData,
    isLoading,
    error,
    mutate,
  } = useSalesReport({
    branchId: selectedBranch,
    fromDate,
    toDate,
  })

  useEffect(() => {
    const fetchBranches = async () => {
      try {
        const response = await fetch("/api/branches")
        const result = await response.json()
        if (result.success) {
          setBranches(result.data)
          console.log("[v0] Loaded branches for sales report filter:", result.data)
        }
      } catch (error) {
        console.error("[v0] Error fetching branches:", error)
      }
    }
    fetchBranches()
  }, [])

  const handleFilter = (e: React.FormEvent) => {
    e.preventDefault()
    console.log("[v0] Filtering sales data with:", { fromDate, toDate, selectedBranch })
    mutate()
  }

  const handleView = (sale: any) => {
    setSelectedSale(sale)
    setIsViewModalOpen(true)
  }

  const handleEdit = (sale: any) => {
    setSelectedSale(sale)
    setEditFormData({
      date: new Date(sale.date).toISOString().split("T")[0],
      branchId: sale.branchId,
      totalCash: sale.totalCash,
      totalCard: sale.totalCard,
      notes: sale.notes || "",
    })
    setIsEditModalOpen(true)
  }

  const handleSaveEdit = async () => {
    try {
      const response = await fetch(`/api/sales/${selectedSale._id}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(editFormData),
      })

      if (response.ok) {
        toast({
          title: "Success",
          description: "Sales record updated successfully",
          variant: "default",
        })
        mutate()
        setIsEditModalOpen(false)
      } else {
        throw new Error("Failed to update sales record")
      }
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update sales record",
        variant: "destructive",
      })
    }
  }

  const handleDelete = async () => {
    try {
      const response = await fetch(`/api/sales/${selectedSale._id}`, {
        method: "DELETE",
      })

      if (response.ok) {
        toast({
          title: "Success",
          description: "Sales record deleted successfully",
          variant: "default",
        })
        mutate()
        setIsDeleteModalOpen(false)
      } else {
        throw new Error("Failed to delete sales record")
      }
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to delete sales record",
        variant: "destructive",
      })
    }
  }

  return (
    <DashboardLayout>
      <div className="p-3 sm:p-4 lg:p-6 xl:p-8 space-y-4 sm:space-y-6">
        {/* Filter Section */}
        <Card className="border-0 shadow-lg bg-gradient-to-br from-blue-50 to-indigo-50 hover:shadow-xl transition-all duration-300">
          <CardHeader className="pb-3 sm:pb-4">
            <CardTitle className="flex items-center gap-2 sm:gap-3 text-lg sm:text-xl">
              <div className="h-8 w-8 sm:h-10 sm:w-10 rounded-xl bg-gradient-to-br from-blue-500 to-indigo-600 flex items-center justify-center shadow-lg">
                <Filter className="h-4 w-4 sm:h-5 sm:w-5 text-white" />
              </div>
              Filter Sales Data
            </CardTitle>
          </CardHeader>
          <CardContent className="px-3 sm:px-4 md:px-6">
            <form onSubmit={handleFilter} className="space-y-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4">
                <div className="space-y-2">
                  <Label htmlFor="fromDate" className="text-xs sm:text-sm font-medium">
                    From Date
                  </Label>
                  <Input
                    id="fromDate"
                    type="date"
                    value={fromDate}
                    onChange={(e) => setFromDate(e.target.value)}
                    className="h-9 sm:h-10 text-xs sm:text-sm"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="toDate" className="text-xs sm:text-sm font-medium">
                    To Date
                  </Label>
                  <Input
                    id="toDate"
                    type="date"
                    value={toDate}
                    onChange={(e) => setToDate(e.target.value)}
                    className="h-9 sm:h-10 text-xs sm:text-sm"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="branch" className="text-xs sm:text-sm font-medium">
                    Branch Name
                  </Label>
                  <Select value={selectedBranch} onValueChange={setSelectedBranch}>
                    <SelectTrigger className="h-9 sm:h-10 text-xs sm:text-sm">
                      <SelectValue placeholder="Select branch" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Branches</SelectItem>
                      {branches.map((branch) => (
                        <SelectItem key={branch._id} value={branch._id}>
                          {branch.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label className="text-xs sm:text-sm font-medium opacity-0 block">Action</Label>
                  <Button type="submit" className="w-full h-9 sm:h-10 bg-blue-600 hover:bg-blue-700 text-xs sm:text-sm">
                    <Filter className="mr-2 h-3 w-3 sm:h-4 sm:w-4" />
                    Filter
                  </Button>
                </div>
              </div>
            </form>
          </CardContent>
        </Card>

        {/* Sales Data Table */}
        <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-gray-50 hover:shadow-xl transition-all duration-300">
          <CardHeader className="pb-3 sm:pb-4">
            <CardTitle className="flex items-center gap-2 sm:gap-3 text-lg sm:text-xl">
              <div className="h-8 w-8 sm:h-10 sm:w-10 rounded-xl bg-gradient-to-br from-green-500 to-emerald-600 flex items-center justify-center shadow-lg">
                <BarChart3 className="h-4 w-4 sm:h-5 sm:w-5 text-white" />
              </div>
              Sales Report List ({salesData.length} records from database)
            </CardTitle>
          </CardHeader>
          <CardContent className="px-3 sm:px-4 md:px-6">
            {isLoading ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-6 w-6 animate-spin mr-2" />
                Loading sales data from database...
              </div>
            ) : error ? (
              <div className="text-center py-8 text-red-600">Error loading sales data: {error}</div>
            ) : (
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow className="hover:bg-transparent">
                      <TableHead className="text-xs sm:text-sm font-semibold">Serial No</TableHead>
                      <TableHead className="text-xs sm:text-sm font-semibold">Date</TableHead>
                      <TableHead className="text-xs sm:text-sm font-semibold">Branch Name</TableHead>
                      <TableHead className="text-xs sm:text-sm font-semibold">Cash Total</TableHead>
                      <TableHead className="text-xs sm:text-sm font-semibold">Card Total</TableHead>
                      <TableHead className="text-xs sm:text-sm font-semibold">Total Sales</TableHead>
                      <TableHead className="text-xs sm:text-sm font-semibold">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {salesData.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center py-8 text-gray-500">
                          No sales data found. Add sales data using the "Add Data" page to see records here.
                        </TableCell>
                      </TableRow>
                    ) : (
                      salesData.map((sale, index) => (
                        <TableRow key={sale._id} className="hover:bg-gray-50/50">
                          <TableCell className="text-xs sm:text-sm font-medium">{index + 1}</TableCell>
                          <TableCell className="text-xs sm:text-sm">
                            {new Date(sale.date).toLocaleDateString()}
                          </TableCell>
                          <TableCell className="text-xs sm:text-sm">{sale.branchName}</TableCell>
                          <TableCell className="text-xs sm:text-sm font-medium">
                            ${sale.totalCash.toLocaleString()}
                          </TableCell>
                          <TableCell className="text-xs sm:text-sm font-medium">
                            ${sale.totalCard.toLocaleString()}
                          </TableCell>
                          <TableCell className="text-xs sm:text-sm font-bold text-green-600">
                            ${sale.totalSales.toLocaleString()}
                          </TableCell>
                          <TableCell>
                            <div className="flex flex-col sm:flex-row gap-1 sm:gap-2">
                              <Button
                                size="sm"
                                variant="outline"
                                className="h-7 sm:h-8 px-2 sm:px-3 text-xs bg-transparent"
                                onClick={() => handleEdit(sale)}
                              >
                                <Edit className="h-3 w-3 sm:mr-1" />
                                <span className="hidden sm:inline">Edit</span>
                              </Button>
                              <Button
                                size="sm"
                                variant="outline"
                                className="h-7 sm:h-8 px-2 sm:px-3 text-xs bg-transparent"
                                onClick={() => handleView(sale)}
                              >
                                <Eye className="h-3 w-3 sm:mr-1" />
                                <span className="hidden sm:inline">View</span>
                              </Button>
                              <Button
                                size="sm"
                                variant="outline"
                                className="h-7 sm:h-8 px-2 sm:px-3 text-xs text-red-600 hover:text-red-700 bg-transparent"
                                onClick={() => {
                                  setSelectedSale(sale)
                                  setIsDeleteModalOpen(true)
                                }}
                              >
                                <Trash2 className="h-3 w-3 sm:mr-1" />
                                <span className="hidden sm:inline">Delete</span>
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>

        <Dialog open={isViewModalOpen} onOpenChange={setIsViewModalOpen}>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>View Sales Record</DialogTitle>
              <DialogDescription>Complete details of the selected sales record.</DialogDescription>
            </DialogHeader>
            {selectedSale && (
              <div className="grid grid-cols-2 gap-4 py-4">
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Date</Label>
                  <p className="text-sm">{new Date(selectedSale.date).toLocaleDateString()}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Branch</Label>
                  <p className="text-sm">{selectedSale.branchName}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Cash Total</Label>
                  <p className="text-sm font-semibold text-green-600">${selectedSale.totalCash.toLocaleString()}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Card Total</Label>
                  <p className="text-sm font-semibold text-green-600">${selectedSale.totalCard.toLocaleString()}</p>
                </div>
                <div className="col-span-2 space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Total Sales</Label>
                  <p className="text-lg font-bold text-green-600">${selectedSale.totalSales.toLocaleString()}</p>
                </div>
                {selectedSale.notes && (
                  <div className="col-span-2 space-y-2">
                    <Label className="text-sm font-medium text-gray-600">Notes</Label>
                    <p className="text-sm">{selectedSale.notes}</p>
                  </div>
                )}
              </div>
            )}
          </DialogContent>
        </Dialog>

        <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Edit Sales Record</DialogTitle>
              <DialogDescription>Update the sales record details.</DialogDescription>
            </DialogHeader>
            <div className="grid grid-cols-2 gap-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="edit-date">Date</Label>
                <Input
                  id="edit-date"
                  type="date"
                  value={editFormData.date || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, date: e.target.value })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-branch">Branch</Label>
                <Select
                  value={editFormData.branchId || ""}
                  onValueChange={(value) => setEditFormData({ ...editFormData, branchId: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select branch" />
                  </SelectTrigger>
                  <SelectContent>
                    {branches.map((branch) => (
                      <SelectItem key={branch._id} value={branch._id}>
                        {branch.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-cash">Cash Total</Label>
                <Input
                  id="edit-cash"
                  type="number"
                  value={editFormData.totalCash || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, totalCash: Number(e.target.value) })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-card">Card Total</Label>
                <Input
                  id="edit-card"
                  type="number"
                  value={editFormData.totalCard || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, totalCard: Number(e.target.value) })}
                />
              </div>
              <div className="col-span-2 space-y-2">
                <Label htmlFor="edit-notes">Notes</Label>
                <Input
                  id="edit-notes"
                  value={editFormData.notes || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, notes: e.target.value })}
                  placeholder="Optional notes"
                />
              </div>
            </div>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsEditModalOpen(false)}>
                Cancel
              </Button>
              <Button onClick={handleSaveEdit}>Save Changes</Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <Dialog open={isDeleteModalOpen} onOpenChange={setIsDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Sales Record</DialogTitle>
              <DialogDescription>
                Are you sure you want to delete this sales record? This action cannot be undone.
              </DialogDescription>
            </DialogHeader>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsDeleteModalOpen(false)}>
                Cancel
              </Button>
              <Button variant="destructive" onClick={handleDelete}>
                Delete
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  )
}
