"use client"

import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { BarChart3, FileText, DollarSign, CreditCard, Banknote, RefreshCw, Loader2 } from "lucide-react"
import {
  Breadcrumb,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbList,
  BreadcrumbPage,
  BreadcrumbSeparator,
} from "@/components/ui/breadcrumb"
import DashboardLayout from "@/components/dashboard-layout"
import { useBranches } from "@/hooks/use-dashboard-data"
import { useSalesReport, useExpenseReport, useIncomeStatement, useSalaryReport } from "@/hooks/use-reports-data"

export default function ReportsPage() {
  const [filters, setFilters] = useState({
    branchId: "all",
    fromDate: new Date().toISOString().split("T")[0],
    toDate: new Date().toISOString().split("T")[0],
  })

  const { branches, isLoading: branchesLoading } = useBranches()
  const { data: salesData, isLoading: salesLoading, error: salesError, mutate: mutateSales } = useSalesReport(filters)
  const {
    data: expenseData,
    isLoading: expenseLoading,
    error: expenseError,
    mutate: mutateExpenses,
  } = useExpenseReport(filters)
  const { data: incomeStatement, isLoading: incomeLoading, error: incomeError } = useIncomeStatement(filters)
  const {
    data: salaryData,
    isLoading: salaryLoading,
    error: salaryError,
    mutate: mutateSalary,
  } = useSalaryReport(filters)

  const handleRefresh = () => {
    mutateSales()
    mutateExpenses()
    mutateSalary()
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("pl-PL", {
      style: "currency",
      currency: "PLN",
    }).format(amount)
  }

  const getSelectedBranchName = () => {
    if (filters.branchId === "all") return "All Branches"
    const branch = branches.find((b) => b._id === filters.branchId)
    return branch?.name || "Unknown Branch"
  }

  return (
    <DashboardLayout>
      <div className="p-3 sm:p-4 lg:p-6 xl:p-8">
        {/* Breadcrumb */}
        <div className="mb-4 sm:mb-6">
          <Breadcrumb>
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink href="/">Dashboard</BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbPage>Reports</BreadcrumbPage>
              </BreadcrumbItem>
            </BreadcrumbList>
          </Breadcrumb>
        </div>

        {/* Header */}
        <div className="mb-6 sm:mb-8">
          <div className="flex items-center gap-3 mb-2">
            <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center shadow-lg">
              <FileText className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
            </div>
            <div>
              <h1 className="text-xl sm:text-2xl md:text-3xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                Financial Reports
              </h1>
              <p className="text-sm sm:text-base text-muted-foreground">
                Comprehensive business analytics and reporting dashboard
              </p>
            </div>
          </div>
        </div>

        {/* Filters */}
        <Card className="border-0 shadow-lg bg-white/80 backdrop-blur-sm ring-1 ring-gray-200/50 mb-6">
          <CardContent className="px-3 sm:px-6 py-6">
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6">
              <div className="space-y-2">
                <Label htmlFor="fromDate" className="text-xs sm:text-sm font-medium text-gray-700">
                  From Date
                </Label>
                <Input
                  id="fromDate"
                  type="date"
                  value={filters.fromDate}
                  onChange={(e) => setFilters({ ...filters, fromDate: e.target.value })}
                  className="h-10 text-xs sm:text-sm border-gray-200 focus:border-blue-500 focus:ring-blue-500/20"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="toDate" className="text-xs sm:text-sm font-medium text-gray-700">
                  To Date
                </Label>
                <Input
                  id="toDate"
                  type="date"
                  value={filters.toDate}
                  onChange={(e) => setFilters({ ...filters, toDate: e.target.value })}
                  className="h-10 text-xs sm:text-sm border-gray-200 focus:border-blue-500 focus:ring-blue-500/20"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="branch" className="text-xs sm:text-sm font-medium text-gray-700">
                  Branch
                </Label>
                <Select value={filters.branchId} onValueChange={(value) => setFilters({ ...filters, branchId: value })}>
                  <SelectTrigger className="h-10 text-xs sm:text-sm border-gray-200 focus:border-blue-500 focus:ring-blue-500/20">
                    <SelectValue placeholder="Select branch" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Branches</SelectItem>
                    {branches.map((branch) => (
                      <SelectItem key={branch._id} value={branch._id}>
                        {branch.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label className="text-sm font-medium opacity-0 block">Refresh</Label>
                <Button
                  variant="outline"
                  className="w-full h-10 text-xs sm:text-sm bg-transparent"
                  onClick={handleRefresh}
                  disabled={salesLoading || expenseLoading}
                >
                  {salesLoading || expenseLoading ? (
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  ) : (
                    <RefreshCw className="mr-2 h-4 w-4" />
                  )}
                  Refresh
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Reports Tabs */}
        <Tabs defaultValue="income-statement" className="space-y-4 sm:space-y-6">
          <div className="overflow-x-auto">
            <TabsList className="grid w-max grid-cols-6 gap-1 sm:gap-2 p-1">
              <TabsTrigger value="income-statement" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Income Statement
              </TabsTrigger>
              <TabsTrigger value="summary" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Summary
              </TabsTrigger>
              <TabsTrigger value="expense" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Expense
              </TabsTrigger>
              <TabsTrigger value="salary" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Salary
              </TabsTrigger>
              <TabsTrigger value="faktura" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Faktura
              </TabsTrigger>
              <TabsTrigger value="bank-data" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Bank Data
              </TabsTrigger>
            </TabsList>
          </div>

          {/* Income Statement Tab */}
          <TabsContent value="income-statement" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-blue-50/30">
              <CardHeader className="text-center pb-4 sm:pb-6">
                <CardTitle className="text-xl sm:text-2xl md:text-3xl font-bold">Diamond Kebab</CardTitle>
                <div className="space-y-1 sm:space-y-2 text-sm sm:text-base text-muted-foreground">
                  <p className="text-lg sm:text-xl font-semibold">{getSelectedBranchName()}</p>
                  <p className="text-base sm:text-lg font-medium">Income Statement</p>
                  <p>For the Period Of</p>
                  <p>
                    Branch Report ({filters.fromDate} to {filters.toDate})
                  </p>
                </div>
              </CardHeader>
              <CardContent className="px-3 sm:px-6">
                {incomeLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading income statement...
                  </div>
                ) : incomeError ? (
                  <div className="text-center py-8 text-red-600">Error loading income statement: {incomeError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-gray-50">
                          <th className="border border-gray-300 px-2 sm:px-4 py-2 text-left font-semibold">Category</th>
                          <th className="border border-gray-300 px-2 sm:px-4 py-2 text-left font-semibold">Amount</th>
                          <th className="border border-gray-300 px-2 sm:px-4 py-2 text-center font-semibold">Totals</th>
                        </tr>
                      </thead>
                      <tbody>
                        {/* Sales Section */}
                        <tr className="bg-blue-50">
                          <td colSpan={2} className="border border-gray-300 px-2 sm:px-4 py-2 font-bold">
                            <DollarSign className="inline mr-2 h-4 w-4" />
                            Sales
                          </td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2 text-center font-bold">
                            Total Sales: {formatCurrency(incomeStatement.sales.totalSales)}
                          </td>
                        </tr>
                        <tr>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">Cash in Shop</td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">
                            {formatCurrency(incomeStatement.sales.cashInShop)}
                          </td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2"></td>
                        </tr>
                        <tr>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">Cash Delivery</td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">
                            {formatCurrency(incomeStatement.sales.cashDelivery)}
                          </td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2"></td>
                        </tr>
                        <tr>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">Card in Shop</td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">
                            {formatCurrency(incomeStatement.sales.cardInShop)}
                          </td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2"></td>
                        </tr>
                        <tr>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">Card Delivery</td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2">
                            {formatCurrency(incomeStatement.sales.cardDelivery)}
                          </td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2"></td>
                        </tr>

                        {/* Expenses Section */}
                        <tr className="bg-red-50">
                          <td colSpan={2} className="border border-gray-300 px-2 sm:px-4 py-2 font-bold">
                            <CreditCard className="inline mr-2 h-4 w-4" />
                            Expenses
                          </td>
                          <td className="border border-gray-300 px-2 sm:px-4 py-2 text-center font-bold">
                            Total Expenses: {formatCurrency(incomeStatement.totalExpenses)}
                          </td>
                        </tr>
                        {Object.entries(incomeStatement.expenses).map(([category, amount]) => (
                          <tr key={category}>
                            <td className="border border-gray-300 px-2 sm:px-4 py-2 capitalize">
                              {category.replace("-", " ")}
                            </td>
                            <td className="border border-gray-300 px-2 sm:px-4 py-2">{formatCurrency(amount)}</td>
                            <td className="border border-gray-300 px-2 sm:px-4 py-2"></td>
                          </tr>
                        ))}

                        {/* Profit/Loss */}
                        <tr className={`${incomeStatement.profitLoss >= 0 ? "bg-green-50" : "bg-red-50"}`}>
                          <td colSpan={2} className="border border-gray-300 px-2 sm:px-4 py-2 font-bold">
                            <Banknote className="inline mr-2 h-4 w-4" />
                            {incomeStatement.profitLoss >= 0 ? "Profit" : "Loss"}
                          </td>
                          <td
                            className={`border border-gray-300 px-2 sm:px-4 py-2 text-center font-bold ${incomeStatement.profitLoss >= 0 ? "text-green-600" : "text-red-600"}`}
                          >
                            {incomeStatement.profitLoss >= 0 ? "Profit" : "Loss"}:{" "}
                            {formatCurrency(Math.abs(incomeStatement.profitLoss))}
                          </td>
                        </tr>
                      </tbody>
                    </table>
                  </div>
                )}

                {/* Signature Section */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-6 sm:mt-8">
                  {["Prepared by", "Checked by", "Manager by"].map((role) => (
                    <div key={role} className="border border-gray-300 p-4 text-center">
                      <div className="mb-4">
                        <div className="w-24 h-16 mx-auto bg-gray-100 border rounded flex items-center justify-center text-xs text-gray-500">
                          Signature
                        </div>
                      </div>
                      <hr className="my-2" />
                      <p className="font-semibold text-sm">{role}</p>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Summary Tab */}
          <TabsContent value="summary" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-green-50/30">
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-lg sm:text-xl">
                  <div className="h-8 w-8 rounded-lg bg-gradient-to-br from-green-500 to-emerald-600 flex items-center justify-center">
                    <BarChart3 className="h-4 w-4 text-white" />
                  </div>
                  Summary Report
                </CardTitle>
              </CardHeader>
              <CardContent>
                {salesLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading sales summary...
                  </div>
                ) : salesError ? (
                  <div className="text-center py-8 text-red-600">Error loading sales data: {salesError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-green-50">
                          <th className="border border-gray-300 px-2 py-2 text-left">Serial No</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Date</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Branch</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash in shop</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash delivery</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash wolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash bolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash pyszne</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash uber eats</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash glovo</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card in shop</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card delivery</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card wolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card bolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card pyszne</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card uber eats</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card glovo</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Total</th>
                        </tr>
                      </thead>
                      <tbody>
                        {salesData.length === 0 ? (
                          <tr>
                            <td colSpan={18} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          salesData.map((sale, index) => (
                            <tr key={sale._id}>
                              <td className="border border-gray-300 px-2 py-2">{index + 1}</td>
                              <td className="border border-gray-300 px-2 py-2">
                                {new Date(sale.date).toLocaleDateString()}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{sale.branchName}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashInShop)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashDelivery)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashWolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashBolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashPyszne)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashUberEats)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashGlovo)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardInShop)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardDelivery)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardWolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardBolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardPyszne)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardUberEats)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardGlovo)}</td>
                              <td className="border border-gray-300 px-2 py-2 font-semibold">
                                {formatCurrency(sale.totalSales)}
                              </td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Expense Tab */}
          <TabsContent value="expense" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-red-50/30">
              <CardHeader className="text-center">
                <CardTitle className="text-xl sm:text-2xl font-bold">Diamond Kebab</CardTitle>
                <div className="space-y-1 text-sm sm:text-base text-muted-foreground">
                  <p className="text-lg font-semibold">{getSelectedBranchName()}</p>
                  <p className="text-base font-medium">Expense Report</p>
                  <p>For the Period Of</p>
                  <p>
                    Branch Report ({filters.fromDate} to {filters.toDate})
                  </p>
                </div>
              </CardHeader>
              <CardContent>
                {expenseLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading expense data...
                  </div>
                ) : expenseError ? (
                  <div className="text-center py-8 text-red-600">Error loading expense data: {expenseError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-red-50">
                          <th className="border border-gray-300 px-2 py-2">Date</th>
                          <th className="border border-gray-300 px-2 py-2">Category</th>
                          <th className="border border-gray-300 px-2 py-2">Description</th>
                          <th className="border border-gray-300 px-2 py-2">Payment Method</th>
                          <th className="border border-gray-300 px-2 py-2">Amount</th>
                          <th className="border border-gray-300 px-2 py-2">Faktura No</th>
                          <th className="border border-gray-300 px-2 py-2">Notes</th>
                        </tr>
                      </thead>
                      <tbody>
                        {expenseData.length === 0 ? (
                          <tr>
                            <td colSpan={7} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No expense data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          expenseData.map((expense) => (
                            <tr key={expense._id}>
                              <td className="border border-gray-300 px-2 py-2">
                                {new Date(expense.date).toLocaleDateString()}
                              </td>
                              <td className="border border-gray-300 px-2 py-2 capitalize">
                                {expense.category.replace("-", " ")}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{expense.description}</td>
                              <td className="border border-gray-300 px-2 py-2 capitalize">{expense.paymentMethod}</td>
                              <td className="border border-gray-300 px-2 py-2 font-semibold">
                                {formatCurrency(expense.amount)}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{expense.fakturaNo || "-"}</td>
                              <td className="border border-gray-300 px-2 py-2">{expense.notes || "-"}</td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Salary Tab */}
          <TabsContent value="salary" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg">
              <CardHeader>
                <CardTitle>Salary Report</CardTitle>
                <p className="text-sm text-muted-foreground">
                  Salary expenses for {getSelectedBranchName()} from {filters.fromDate} to {filters.toDate}
                </p>
              </CardHeader>
              <CardContent>
                {salaryLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading salary data...
                  </div>
                ) : salaryError ? (
                  <div className="text-center py-8 text-red-600">Error loading salary data: {salaryError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-blue-50">
                          <th className="border border-gray-300 px-2 py-2">Date</th>
                          <th className="border border-gray-300 px-2 py-2">Employee</th>
                          <th className="border border-gray-300 px-2 py-2">Description</th>
                          <th className="border border-gray-300 px-2 py-2">Amount</th>
                          <th className="border border-gray-300 px-2 py-2">Payment Method</th>
                          <th className="border border-gray-300 px-2 py-2">Notes</th>
                        </tr>
                      </thead>
                      <tbody>
                        {salaryData.length === 0 ? (
                          <tr>
                            <td colSpan={6} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No salary data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          salaryData.map((salary) => (
                            <tr key={salary._id}>
                              <td className="border border-gray-300 px-2 py-2">
                                {new Date(salary.date).toLocaleDateString()}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{salary.subcategory || "N/A"}</td>
                              <td className="border border-gray-300 px-2 py-2">{salary.description}</td>
                              <td className="border border-gray-300 px-2 py-2 font-semibold">
                                {formatCurrency(salary.amount)}
                              </td>
                              <td className="border border-gray-300 px-2 py-2 capitalize">{salary.paymentMethod}</td>
                              <td className="border border-gray-300 px-2 py-2">{salary.notes || "-"}</td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Faktura Tab */}
          <TabsContent value="faktura" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg">
              <CardHeader>
                <CardTitle>Faktura Report</CardTitle>
                <p className="text-sm text-muted-foreground">
                  Expenses with faktura numbers for {getSelectedBranchName()} from {filters.fromDate} to{" "}
                  {filters.toDate}
                </p>
              </CardHeader>
              <CardContent>
                {expenseLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading faktura data...
                  </div>
                ) : expenseError ? (
                  <div className="text-center py-8 text-red-600">Error loading faktura data: {expenseError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-purple-50">
                          <th className="border border-gray-300 px-2 py-2">Date</th>
                          <th className="border border-gray-300 px-2 py-2">Faktura No</th>
                          <th className="border border-gray-300 px-2 py-2">Category</th>
                          <th className="border border-gray-300 px-2 py-2">Description</th>
                          <th className="border border-gray-300 px-2 py-2">Amount</th>
                          <th className="border border-gray-300 px-2 py-2">Payment Date</th>
                        </tr>
                      </thead>
                      <tbody>
                        {expenseData.filter((expense) => expense.fakturaNo).length === 0 ? (
                          <tr>
                            <td colSpan={6} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No faktura data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          expenseData
                            .filter((expense) => expense.fakturaNo)
                            .map((expense) => (
                              <tr key={expense._id}>
                                <td className="border border-gray-300 px-2 py-2">
                                  {new Date(expense.date).toLocaleDateString()}
                                </td>
                                <td className="border border-gray-300 px-2 py-2 font-mono">{expense.fakturaNo}</td>
                                <td className="border border-gray-300 px-2 py-2 capitalize">
                                  {expense.category.replace("-", " ")}
                                </td>
                                <td className="border border-gray-300 px-2 py-2">{expense.description}</td>
                                <td className="border border-gray-300 px-2 py-2 font-semibold">
                                  {formatCurrency(expense.amount)}
                                </td>
                                <td className="border border-gray-300 px-2 py-2">
                                  {expense.paymentDate ? new Date(expense.paymentDate).toLocaleDateString() : "-"}
                                </td>
                              </tr>
                            ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Bank Data Tab */}
          <TabsContent value="bank-data" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg">
              <CardHeader>
                <CardTitle>Bank Data Report</CardTitle>
                <p className="text-sm text-muted-foreground">
                  Card transactions for {getSelectedBranchName()} from {filters.fromDate} to {filters.toDate}
                </p>
              </CardHeader>
              <CardContent>
                {salesLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading bank data...
                  </div>
                ) : salesError ? (
                  <div className="text-center py-8 text-red-600">Error loading bank data: {salesError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-indigo-50">
                          <th className="border border-gray-300 px-2 py-2">Date</th>
                          <th className="border border-gray-300 px-2 py-2">Branch</th>
                          <th className="border border-gray-300 px-2 py-2">Card in Shop</th>
                          <th className="border border-gray-300 px-2 py-2">Card Delivery</th>
                          <th className="border border-gray-300 px-2 py-2">Card Wolt</th>
                          <th className="border border-gray-300 px-2 py-2">Card Bolt</th>
                          <th className="border border-gray-300 px-2 py-2">Card Pyszne</th>
                          <th className="border border-gray-300 px-2 py-2">Card Uber Eats</th>
                          <th className="border border-gray-300 px-2 py-2">Card Glovo</th>
                          <th className="border border-gray-300 px-2 py-2">Total Card</th>
                        </tr>
                      </thead>
                      <tbody>
                        {salesData.length === 0 ? (
                          <tr>
                            <td colSpan={10} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No bank data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          salesData.map((sale) => (
                            <tr key={sale._id}>
                              <td className="border border-gray-300 px-2 py-2">
                                {new Date(sale.date).toLocaleDateString()}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{sale.branchName}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardInShop)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardDelivery)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardWolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardBolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardPyszne)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardUberEats)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardGlovo)}</td>
                              <td className="border border-gray-300 px-2 py-2 font-semibold">
                                {formatCurrency(sale.totalCard)}
                              </td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </DashboardLayout>
  )
}
