"use client"

import type React from "react"
import { useState } from "react"
import { useRouter } from "next/navigation"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { ChartContainer } from "@/components/ui/chart"
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  LineChart,
  Line,
  Tooltip,
  Legend,
  ResponsiveContainer,
} from "recharts"
import { TrendingUp, TrendingDown, Building, Search, DollarSign, RefreshCw } from "lucide-react"
import DashboardLayout from "@/components/dashboard-layout"
import { useDashboardData, useBranches } from "@/hooks/use-dashboard-data"

export default function Dashboard() {
  const router = useRouter()
  const [filters, setFilters] = useState({
    dateFrom: "",
    dateTo: "",
    branch: "all",
  })

  const {
    data: dashboardData,
    isLoading,
    error,
    mutate,
  } = useDashboardData({
    branchId: filters.branch,
    fromDate: filters.dateFrom,
    toDate: filters.dateTo,
  })

  const { branches, isLoading: branchesLoading } = useBranches()

  const handleNavigation = (path: string) => {
    router.push(path)
  }

  const handleApplyFilters = () => {
    mutate() // Refresh data with new filters
    handleNavigation("/reports")
  }

  const handleRefresh = () => {
    mutate()
  }

  if (isLoading) {
    return (
      <DashboardLayout>
        <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-50 p-3 sm:p-4 md:p-6 lg:p-8">
          <div className="mx-auto max-w-full sm:max-w-2xl md:max-w-4xl lg:max-w-6xl xl:max-w-7xl">
            <div className="flex items-center justify-center h-64">
              <div className="flex items-center gap-2 text-gray-600">
                <RefreshCw className="h-5 w-5 animate-spin" />
                Loading dashboard data...
              </div>
            </div>
          </div>
        </div>
      </DashboardLayout>
    )
  }

  if (error) {
    return (
      <DashboardLayout>
        <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-50 p-3 sm:p-4 md:p-6 lg:p-8">
          <div className="mx-auto max-w-full sm:max-w-2xl md:max-w-4xl lg:max-w-6xl xl:max-w-7xl">
            <div className="flex items-center justify-center h-64">
              <div className="text-center">
                <p className="text-red-600 mb-4">{error}</p>
                <Button onClick={handleRefresh} variant="outline">
                  <RefreshCw className="h-4 w-4 mr-2" />
                  Try Again
                </Button>
              </div>
            </div>
          </div>
        </div>
      </DashboardLayout>
    )
  }

  return (
    <DashboardLayout>
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-50 p-3 sm:p-4 md:p-6 lg:p-8">
        <div className="mx-auto max-w-full sm:max-w-2xl md:max-w-4xl lg:max-w-6xl xl:max-w-7xl space-y-6 md:space-y-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Dashboard</h1>
              <p className="text-gray-600">Business insights and analytics</p>
            </div>
          </div>

          <Card className="border-0 shadow-lg bg-white/80 backdrop-blur-sm ring-1 ring-gray-200/50">
            <CardContent className="px-3 sm:px-6 py-6">
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4 sm:gap-6">
                <div className="space-y-2">
                  <Label htmlFor="dateFrom" className="text-xs sm:text-sm font-medium text-gray-700">
                    From Date
                  </Label>
                  <Input
                    id="dateFrom"
                    type="date"
                    value={filters.dateFrom}
                    onChange={(e) => setFilters({ ...filters, dateFrom: e.target.value })}
                    className="h-10 text-xs sm:text-sm border-gray-200 focus:border-blue-500 focus:ring-blue-500/20"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="dateTo" className="text-xs sm:text-sm font-medium text-gray-700">
                    To Date
                  </Label>
                  <Input
                    id="dateTo"
                    type="date"
                    value={filters.dateTo}
                    onChange={(e) => setFilters({ ...filters, dateTo: e.target.value })}
                    className="h-10 text-xs sm:text-sm border-gray-200 focus:border-blue-500 focus:ring-blue-500/20"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="branch" className="text-xs sm:text-sm font-medium text-gray-700">
                    Branch
                  </Label>
                  <Select value={filters.branch} onValueChange={(value) => setFilters({ ...filters, branch: value })}>
                    <SelectTrigger className="h-10 text-xs sm:text-sm border-gray-200 focus:border-blue-500 focus:ring-blue-500/20">
                      <SelectValue placeholder="Select branch" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Branches</SelectItem>
                      {branches.map((branch) => (
                        <SelectItem key={branch._id} value={branch._id}>
                          {branch.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium opacity-0 block">Action</Label>
                  <Button
                    className="w-full h-10 text-xs sm:text-sm bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 shadow-lg hover:shadow-xl transition-all duration-200"
                    onClick={handleApplyFilters}
                  >
                    <Search className="mr-2 h-4 w-4" />
                    Apply Filter
                  </Button>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium opacity-0 block">Refresh</Label>
                  <Button
                    variant="outline"
                    className="w-full h-10 text-xs sm:text-sm bg-transparent"
                    onClick={handleRefresh}
                  >
                    <RefreshCw className="mr-2 h-4 w-4" />
                    Refresh
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>

          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 md:gap-6">
            <MetricCard
              title="Total Sales"
              value={dashboardData?.metrics.totalSales.value || 0}
              change={dashboardData?.metrics.totalSales.change || 0}
              trend={dashboardData?.metrics.totalSales.trend || "neutral"}
              icon={<DollarSign className="h-5 w-5" />}
              color="green"
            />
            <MetricCard
              title="Total Expenses"
              value={dashboardData?.metrics.totalExpenses.value || 0}
              change={dashboardData?.metrics.totalExpenses.change || 0}
              trend={dashboardData?.metrics.totalExpenses.trend || "neutral"}
              icon={<TrendingDown className="h-5 w-5" />}
              color="red"
            />
            <MetricCard
              title="Net Profit"
              value={dashboardData?.metrics.netProfit.value || 0}
              change={dashboardData?.metrics.netProfit.change || 0}
              trend={dashboardData?.metrics.netProfit.trend || "neutral"}
              icon={<TrendingUp className="h-5 w-5" />}
              color="blue"
            />
            <MetricCard
              title="Active Branches"
              value={dashboardData?.metrics.activeBranches.value || 0}
              change={dashboardData?.metrics.activeBranches.change || 0}
              trend={dashboardData?.metrics.activeBranches.trend || "neutral"}
              icon={<Building className="h-5 w-5" />}
              color="purple"
            />
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 md:gap-8">
            <Card className="border-0 shadow-lg bg-white/80 backdrop-blur-sm ring-1 ring-gray-200/50 hover:shadow-xl transition-shadow duration-300">
              <CardHeader className="pb-4 bg-gradient-to-r from-green-50 to-emerald-50 rounded-t-lg">
                <CardTitle className="text-sm sm:text-base md:text-lg font-semibold text-gray-800 flex items-center gap-2">
                  <BarChart className="h-5 w-5 text-green-600" />
                  Monthly Sales & Expenses
                </CardTitle>
              </CardHeader>
              <CardContent className="px-3 sm:px-4 md:px-6 py-6">
                <ChartContainer
                  config={{
                    sales: { label: "Sales", color: "hsl(var(--chart-1))" },
                    expenses: { label: "Expenses", color: "hsl(var(--chart-2))" },
                  }}
                  className="h-[200px] sm:h-[240px] md:h-[300px] lg:h-[340px] xl:h-[380px] w-full"
                >
                  <ResponsiveContainer width="100%" height="100%">
                    <BarChart
                      data={dashboardData?.monthlyData || []}
                      margin={{ top: 10, right: 15, left: 0, bottom: 0 }}
                    >
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="month" fontSize={10} />
                      <YAxis fontSize={10} />
                      <Tooltip />
                      <Legend />
                      <Bar dataKey="sales" fill="var(--color-sales)" radius={[2, 2, 0, 0]} />
                      <Bar dataKey="expenses" fill="var(--color-expenses)" radius={[2, 2, 0, 0]} />
                    </BarChart>
                  </ResponsiveContainer>
                </ChartContainer>
              </CardContent>
            </Card>

            <Card className="border-0 shadow-lg bg-white/80 backdrop-blur-sm ring-1 ring-gray-200/50 hover:shadow-xl transition-shadow duration-300">
              <CardHeader className="pb-4 bg-gradient-to-r from-blue-50 to-cyan-50 rounded-t-lg">
                <CardTitle className="text-sm sm:text-base md:text-lg font-semibold text-gray-800 flex items-center gap-2">
                  <TrendingUp className="h-5 w-5 text-blue-600" />
                  Sales Trend
                </CardTitle>
              </CardHeader>
              <CardContent className="px-3 sm:px-4 md:px-6 py-6">
                <ChartContainer
                  config={{
                    sales: { label: "Sales", color: "hsl(var(--chart-1))" },
                  }}
                  className="h-[200px] sm:h-[240px] md:h-[300px] lg:h-[340px] xl:h-[380px] w-full"
                >
                  <ResponsiveContainer width="100%" height="100%">
                    <LineChart
                      data={dashboardData?.monthlyData || []}
                      margin={{ top: 10, right: 15, left: 0, bottom: 0 }}
                    >
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="month" fontSize={10} />
                      <YAxis fontSize={10} />
                      <Tooltip />
                      <Line
                        type="monotone"
                        dataKey="sales"
                        stroke="var(--color-sales)"
                        strokeWidth={2}
                        dot={{ r: 3 }}
                      />
                    </LineChart>
                  </ResponsiveContainer>
                </ChartContainer>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </DashboardLayout>
  )
}

interface MetricCardProps {
  title: string
  value: number
  change: number
  trend: "up" | "down" | "neutral"
  icon: React.ReactNode
  color: "green" | "red" | "blue" | "purple"
}

const colorClasses = {
  green: "bg-green-50 text-green-900",
  red: "bg-red-50 text-red-900",
  blue: "bg-blue-50 text-blue-900",
  purple: "bg-purple-50 text-purple-900",
}

function MetricCard({ title, value, change, trend, icon, color }: MetricCardProps) {
  const formatValue = (num: number) => num.toLocaleString()

  return (
    <Card className={`border-0 shadow-lg ${colorClasses[color]}`}>
      <CardHeader className="pb-4">
        <div className="flex items-center justify-between">
          <div className="text-xs sm:text-sm font-semibold text-gray-700">{title}</div>
          <div className={colorClasses[color].split(" ")[2]}>{icon}</div>
        </div>
        <div className="space-y-2">
          <div className="text-xl sm:text-2xl md:text-3xl font-bold text-gray-900">{formatValue(value)}</div>
          {change !== 0 && (
            <div className="flex items-center text-xs sm:text-sm">
              {trend === "up" ? (
                <TrendingUp className="mr-1 h-3 w-3 text-green-500" />
              ) : trend === "down" ? (
                <TrendingDown className="mr-1 h-3 w-3 text-red-500" />
              ) : null}
              <span
                className={`font-medium ${trend === "up" ? "text-green-600" : trend === "down" ? "text-red-600" : "text-gray-500"}`}
              >
                {change > 0 ? "+" : ""}
                {change}% from last month
              </span>
            </div>
          )}
        </div>
      </CardHeader>
    </Card>
  )
}

interface DataTableProps {
  data: any[]
}

function DataTable({ data }: DataTableProps) {
  if (!data || !Array.isArray(data) || data.length === 0) {
    return <div className="text-center py-8 text-muted-foreground">No data available</div>
  }

  const headers = Object.keys(data[0] || {})

  return (
    <div className="overflow-x-auto">
      <table className="min-w-full divide-y divide-gray-200">
        <thead className="bg-gray-50">
          <tr>
            {headers.map((header) => (
              <th
                key={header}
                scope="col"
                className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider"
              >
                {header.charAt(0).toUpperCase() + header.slice(1)}
              </th>
            ))}
          </tr>
        </thead>
        <tbody className="bg-white divide-y divide-gray-200">
          {data.map((row, index) => (
            <tr key={index}>
              {headers.map((header) => (
                <td key={header} className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                  {row[header]}
                </td>
              ))}
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  )
}
