"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import {
  Breadcrumb,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbList,
  BreadcrumbPage,
  BreadcrumbSeparator,
} from "@/components/ui/breadcrumb"
import { UserPlus, Users, Edit, Trash2, Loader2 } from "lucide-react"
import DashboardLayout from "@/components/dashboard-layout"

interface Owner {
  _id: string
  name: string
  email: string
  phone: string
  role: string
  createdAt: string
}

export default function OwnerManagement() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    password: "",
  })

  const [owners, setOwners] = useState<Owner[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const user = {
    name: "Admin User",
    email: "admin@example.com",
    avatar: "/placeholder.svg?height=40&width=40",
  }

  useEffect(() => {
    fetchOwners()
  }, [])

  const fetchOwners = async () => {
    try {
      console.log("[v0] Fetching owners from database...")
      const response = await fetch("/api/users?role=owner")
      const result = await response.json()
      if (result.success) {
        setOwners(result.data)
        console.log("[v0] Successfully loaded owners:", result.data.length)
      } else {
        setError(result.error || "Failed to fetch owners")
      }
    } catch (error) {
      console.error("[v0] Error fetching owners:", error)
      setError("Network error occurred")
    } finally {
      setLoading(false)
    }
  }

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value,
    })
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    try {
      console.log("[v0] Creating new owner...")
      const response = await fetch("/api/users", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...formData,
          role: "owner",
        }),
      })

      const result = await response.json()
      if (result.success) {
        setOwners([result.data, ...owners])
        setFormData({ name: "", email: "", phone: "", password: "" })
        console.log("[v0] Owner created successfully")
      } else {
        alert("Failed to create owner: " + result.error)
      }
    } catch (error) {
      console.error("[v0] Error creating owner:", error)
      alert("Network error occurred")
    }
  }

  const handleDeleteOwner = async (ownerId: string) => {
    if (!confirm("Are you sure you want to delete this owner?")) return

    try {
      const response = await fetch(`/api/users/${ownerId}`, {
        method: "DELETE",
      })
      const result = await response.json()

      if (result.success) {
        setOwners(owners.filter((owner) => owner._id !== ownerId))
        console.log("[v0] Owner deleted successfully")
      } else {
        alert("Failed to delete owner: " + result.error)
      }
    } catch (error) {
      console.error("[v0] Error deleting owner:", error)
      alert("Network error occurred")
    }
  }

  return (
    <DashboardLayout user={user}>
      <div className="p-3 sm:p-4 md:p-6">
        <div className="max-w-7xl mx-auto space-y-4 sm:space-y-6">
          <Breadcrumb>
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink href="/">Dashboard</BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbPage>Owner Management</BreadcrumbPage>
              </BreadcrumbItem>
            </BreadcrumbList>
          </Breadcrumb>

          {/* Add Owner Form */}
          <Card className="shadow-lg border-0 bg-gradient-to-br from-purple-50 to-pink-50">
            <CardHeader className="pb-4 sm:pb-6">
              <CardTitle className="flex items-center gap-3 text-lg sm:text-xl">
                <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-purple-500 to-pink-600 flex items-center justify-center shadow-lg">
                  <UserPlus className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                </div>
                Add Owner
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4 sm:space-y-6">
              <form onSubmit={handleSubmit} className="space-y-4 sm:space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 sm:gap-6">
                  <div className="space-y-2">
                    <Label htmlFor="name" className="text-sm font-medium">
                      Name <span className="text-red-500">*</span>
                    </Label>
                    <Input
                      id="name"
                      name="name"
                      type="text"
                      placeholder="Enter Name"
                      value={formData.name}
                      onChange={handleInputChange}
                      required
                      className="h-10 sm:h-11"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="email" className="text-sm font-medium">
                      Email <span className="text-red-500">*</span>
                    </Label>
                    <Input
                      id="email"
                      name="email"
                      type="email"
                      placeholder="Enter Email"
                      value={formData.email}
                      onChange={handleInputChange}
                      required
                      className="h-10 sm:h-11"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="phone" className="text-sm font-medium">
                      Phone <span className="text-red-500">*</span>
                    </Label>
                    <Input
                      id="phone"
                      name="phone"
                      type="tel"
                      placeholder="Enter Phone"
                      value={formData.phone}
                      onChange={handleInputChange}
                      required
                      className="h-10 sm:h-11"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="password" className="text-sm font-medium">
                      Password <span className="text-red-500">*</span>
                    </Label>
                    <Input
                      id="password"
                      name="password"
                      type="password"
                      placeholder="Enter Password"
                      value={formData.password}
                      onChange={handleInputChange}
                      required
                      className="h-10 sm:h-11"
                    />
                  </div>
                </div>
                <div className="flex justify-end">
                  <Button
                    type="submit"
                    className="bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700 text-white px-6 sm:px-8 py-2 sm:py-3 text-sm sm:text-base font-medium shadow-lg"
                  >
                    <UserPlus className="mr-2 h-4 w-4" />
                    Create Owner
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>

          {/* Owner List */}
          <Card className="shadow-lg border-0">
            <CardHeader className="pb-4 sm:pb-6">
              <CardTitle className="flex items-center gap-3 text-lg sm:text-xl">
                <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center shadow-lg">
                  <Users className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                </div>
                Owner List ({owners.length} owners)
              </CardTitle>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="h-6 w-6 animate-spin mr-2" />
                  Loading owners from database...
                </div>
              ) : error ? (
                <div className="text-center py-8 text-red-600">Error loading owners: {error}</div>
              ) : (
                <div className="overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="text-xs sm:text-sm">Serial No</TableHead>
                        <TableHead className="text-xs sm:text-sm">Name</TableHead>
                        <TableHead className="text-xs sm:text-sm">Email</TableHead>
                        <TableHead className="text-xs sm:text-sm">Phone</TableHead>
                        <TableHead className="text-xs sm:text-sm">Created At</TableHead>
                        <TableHead className="text-xs sm:text-sm">Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {owners.length === 0 ? (
                        <TableRow>
                          <TableCell colSpan={6} className="text-center py-8 text-gray-500">
                            No owners found in database
                          </TableCell>
                        </TableRow>
                      ) : (
                        owners.map((owner, index) => (
                          <TableRow key={owner._id}>
                            <TableCell className="text-xs sm:text-sm">{index + 1}</TableCell>
                            <TableCell className="text-xs sm:text-sm font-medium">{owner.name}</TableCell>
                            <TableCell className="text-xs sm:text-sm">{owner.email}</TableCell>
                            <TableCell className="text-xs sm:text-sm">{owner.phone}</TableCell>
                            <TableCell className="text-xs sm:text-sm">
                              {new Date(owner.createdAt).toLocaleDateString()}
                            </TableCell>
                            <TableCell>
                              <div className="flex flex-col sm:flex-row gap-1 sm:gap-2">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 sm:h-8 px-2 sm:px-3 text-xs bg-transparent"
                                >
                                  <Edit className="h-3 w-3 mr-1" />
                                  Edit
                                </Button>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 sm:h-8 px-2 sm:px-3 text-xs text-red-600 hover:text-red-700 bg-transparent"
                                  onClick={() => handleDeleteOwner(owner._id)}
                                >
                                  <Trash2 className="h-3 w-3 mr-1" />
                                  Delete
                                </Button>
                              </div>
                            </TableCell>
                          </TableRow>
                        ))
                      )}
                    </TableBody>
                  </Table>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  )
}
