"use client"
import { useState, useEffect } from "react"
import DashboardLayout from "@/components/dashboard-layout"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Edit, Eye, Trash2, Package, Loader2, Filter } from "lucide-react"
import { useBranches } from "@/hooks/use-dashboard-data"
import { useToast } from "@/hooks/use-toast"

interface InventoryItem {
  _id: string
  date: string
  branchName: string
  type: "daily-consumption" | "stock-level"
  kurczak: number
  wolowina: number
  kraftowe: number
  majonez: number
  ketchup: number
  smabal: number
  yogurt: number
  pita20cm: number
  pita30cm: number
  gas: number
  notes?: string
}

export default function InventoryReportPage() {
  const [consumptionData, setConsumptionData] = useState<InventoryItem[]>([])
  const [inventoryData, setInventoryData] = useState<InventoryItem[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const [selectedItem, setSelectedItem] = useState<InventoryItem | null>(null)
  const [isViewModalOpen, setIsViewModalOpen] = useState(false)
  const [isEditModalOpen, setIsEditModalOpen] = useState(false)
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false)
  const [editFormData, setEditFormData] = useState<any>({})
  const { toast } = useToast()

  const [filters, setFilters] = useState({
    fromDate: "",
    toDate: "",
    branchId: "all",
  })

  const { branches } = useBranches()

  useEffect(() => {
    fetchInventoryData()
  }, [filters])

  const fetchInventoryData = async () => {
    try {
      console.log("[v0] Fetching inventory data with filters:", filters)
      setLoading(true)

      const queryParams = new URLSearchParams()
      if (filters.fromDate) queryParams.append("fromDate", filters.fromDate)
      if (filters.toDate) queryParams.append("toDate", filters.toDate)
      if (filters.branchId && filters.branchId !== "all") queryParams.append("branchId", filters.branchId)

      // Fetch consumption data (daily-consumption)
      const consumptionParams = new URLSearchParams(queryParams)
      consumptionParams.append("type", "daily-consumption")
      const consumptionResponse = await fetch(`/api/inventory?${consumptionParams}`)
      const consumptionResult = await consumptionResponse.json()

      // Fetch stock data (stock-level)
      const stockParams = new URLSearchParams(queryParams)
      stockParams.append("type", "stock-level")
      const stockResponse = await fetch(`/api/inventory?${stockParams}`)
      const stockResult = await stockResponse.json()

      if (consumptionResult.success && stockResult.success) {
        setConsumptionData(consumptionResult.data)
        setInventoryData(stockResult.data)
        console.log("[v0] Successfully loaded filtered inventory data:", {
          consumption: consumptionResult.data.length,
          stock: stockResult.data.length,
        })
      } else {
        setError("Failed to fetch inventory data")
      }
    } catch (error) {
      console.error("[v0] Error fetching inventory data:", error)
      setError("Network error occurred")
    } finally {
      setLoading(false)
    }
  }

  const handleFilterChange = (key: string, value: string) => {
    setFilters((prev) => ({
      ...prev,
      [key]: value,
    }))
  }

  const clearFilters = () => {
    setFilters({
      fromDate: "",
      toDate: "",
      branchId: "all",
    })
  }

  const handleView = (item: InventoryItem) => {
    setSelectedItem(item)
    setIsViewModalOpen(true)
  }

  const handleEdit = (item: InventoryItem) => {
    setSelectedItem(item)
    setEditFormData({
      date: new Date(item.date).toISOString().split("T")[0],
      branchId: item.branchId || "",
      type: item.type,
      kurczak: item.kurczak,
      wolowina: item.wolowina,
      kraftowe: item.kraftowe,
      majonez: item.majonez,
      ketchup: item.ketchup,
      smabal: item.smabal || 0,
      yogurt: item.yogurt || 0,
      pita20cm: item.pita20cm || 0,
      pita30cm: item.pita30cm || 0,
      gas: item.gas,
      notes: item.notes || "",
    })
    setIsEditModalOpen(true)
  }

  const handleSaveEdit = async () => {
    try {
      const response = await fetch(`/api/inventory/${selectedItem?._id}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(editFormData),
      })

      if (response.ok) {
        toast({
          title: "Success",
          description: "Inventory record updated successfully",
          variant: "default",
        })
        fetchInventoryData()
        setIsEditModalOpen(false)
      } else {
        throw new Error("Failed to update inventory record")
      }
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update inventory record",
        variant: "destructive",
      })
    }
  }

  const handleDelete = async () => {
    try {
      const response = await fetch(`/api/inventory/${selectedItem?._id}`, {
        method: "DELETE",
      })

      if (response.ok) {
        toast({
          title: "Success",
          description: "Inventory record deleted successfully",
          variant: "default",
        })
        fetchInventoryData()
        setIsDeleteModalOpen(false)
      } else {
        throw new Error("Failed to delete inventory record")
      }
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to delete inventory record",
        variant: "destructive",
      })
    }
  }

  return (
    <DashboardLayout user={{ name: "John Doe", email: "john@example.com" }}>
      <div className="p-3 sm:p-4 lg:p-8">
        <div className="mb-6 sm:mb-8">
          <div className="flex items-center gap-2 sm:gap-3 mb-2">
            <div className="h-8 w-8 sm:h-10 sm:w-10 rounded-xl bg-gradient-to-br from-green-500 to-emerald-600 flex items-center justify-center shadow-lg">
              <Package className="h-4 w-4 sm:h-5 sm:w-5 text-white" />
            </div>
            <h1 className="text-xl sm:text-2xl lg:text-3xl font-bold bg-gradient-to-r from-gray-900 to-gray-600 bg-clip-text text-transparent">
              Inventory Management
            </h1>
          </div>
          <p className="text-sm sm:text-base text-muted-foreground">
            Track daily consumption and current inventory levels from database records
          </p>
        </div>

        <Card className="mb-6 border-0 shadow-lg bg-gradient-to-br from-blue-50 to-indigo-50/50">
          <CardHeader className="pb-4">
            <CardTitle className="flex items-center gap-2 text-lg">
              <Filter className="h-5 w-5" />
              Filter Inventory Data
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
              <div className="space-y-2">
                <Label htmlFor="from-date" className="text-sm font-medium">
                  From Date
                </Label>
                <Input
                  id="from-date"
                  type="date"
                  value={filters.fromDate}
                  onChange={(e) => handleFilterChange("fromDate", e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="to-date" className="text-sm font-medium">
                  To Date
                </Label>
                <Input
                  id="to-date"
                  type="date"
                  value={filters.toDate}
                  onChange={(e) => handleFilterChange("toDate", e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="branch-filter" className="text-sm font-medium">
                  Branch
                </Label>
                <Select value={filters.branchId} onValueChange={(value) => handleFilterChange("branchId", value)}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select branch" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Branches</SelectItem>
                    {branches?.map((branch) => (
                      <SelectItem key={branch._id} value={branch._id}>
                        {branch.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="flex items-end">
                <Button onClick={clearFilters} variant="outline" className="w-full bg-transparent">
                  Clear Filters
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-slate-50/50">
          <CardHeader className="pb-4 sm:pb-6">
            <CardTitle className="text-lg sm:text-xl">
              Inventory List ({consumptionData.length + inventoryData.length} total records from database)
            </CardTitle>
          </CardHeader>
          <CardContent className="px-3 sm:px-4 md:px-6">
            {loading ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-6 w-6 animate-spin mr-2" />
                Loading inventory data from database...
              </div>
            ) : error ? (
              <div className="text-center py-8 text-red-600">Error loading inventory data: {error}</div>
            ) : (
              <Tabs defaultValue="daily-consumption" className="space-y-4 sm:space-y-6">
                <TabsList className="grid w-full grid-cols-2 h-9 sm:h-10">
                  <TabsTrigger value="daily-consumption" className="text-xs sm:text-sm">
                    Daily Consumption ({consumptionData.length} records)
                  </TabsTrigger>
                  <TabsTrigger value="inventory" className="text-xs sm:text-sm">
                    Add Inventory ({inventoryData.length} records)
                  </TabsTrigger>
                </TabsList>

                <TabsContent value="daily-consumption" className="space-y-4">
                  <div className="rounded-lg border overflow-hidden">
                    <div className="overflow-x-auto">
                      <Table>
                        <TableHeader>
                          <TableRow className="bg-muted/50">
                            <TableHead className="text-xs sm:text-sm font-semibold">Serial No</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Date</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Branch Name</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Kurczak</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Wolowina</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Kraftowe</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Majonez</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Ketchup</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Gas</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Action</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {consumptionData.length === 0 ? (
                            <TableRow>
                              <TableCell colSpan={10} className="text-center py-8 text-gray-500">
                                No consumption data found. Add inventory data using the "Add Data" page to see records
                                here.
                              </TableCell>
                            </TableRow>
                          ) : (
                            consumptionData.map((item, index) => (
                              <TableRow key={item._id} className="hover:bg-muted/30">
                                <TableCell className="text-xs sm:text-sm font-medium">{index + 1}</TableCell>
                                <TableCell className="text-xs sm:text-sm">
                                  {new Date(item.date).toLocaleDateString()}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm">{item.branchName}</TableCell>
                                <TableCell className="text-xs sm:text-sm font-medium text-red-600">
                                  -{item.kurczak}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-medium text-red-600">
                                  -{item.wolowina}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-medium text-red-600">
                                  -{item.kraftowe}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-medium text-red-600">
                                  -{item.majonez}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-medium text-red-600">
                                  -{item.ketchup}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-medium text-red-600">
                                  -{item.gas}
                                </TableCell>
                                <TableCell>
                                  <div className="flex flex-col sm:flex-row gap-1 sm:gap-2">
                                    <Button
                                      size="sm"
                                      variant="outline"
                                      className="h-7 px-2 text-xs bg-transparent"
                                      onClick={() => handleEdit(item)}
                                    >
                                      <Edit className="h-3 w-3 mr-1" />
                                      Edit
                                    </Button>
                                    <Button
                                      size="sm"
                                      variant="outline"
                                      className="h-7 px-2 text-xs bg-transparent"
                                      onClick={() => handleView(item)}
                                    >
                                      <Eye className="h-3 w-3 mr-1" />
                                      View
                                    </Button>
                                    <Button
                                      size="sm"
                                      variant="outline"
                                      className="h-7 px-2 text-xs text-red-600 hover:text-red-700 bg-transparent"
                                      onClick={() => {
                                        setSelectedItem(item)
                                        setIsDeleteModalOpen(true)
                                      }}
                                    >
                                      <Trash2 className="h-3 w-3 mr-1" />
                                      Delete
                                    </Button>
                                  </div>
                                </TableCell>
                              </TableRow>
                            ))
                          )}
                        </TableBody>
                      </Table>
                    </div>
                  </div>
                </TabsContent>

                <TabsContent value="inventory" className="space-y-4">
                  <div className="rounded-lg border overflow-hidden">
                    <div className="overflow-x-auto">
                      <Table>
                        <TableHeader>
                          <TableRow className="bg-muted/50">
                            <TableHead className="text-xs sm:text-sm font-semibold">Serial No</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Date</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Branch Name</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Kurczak</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Wolowina</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Kraftowe</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Majonez</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Ketchup</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Gas</TableHead>
                            <TableHead className="text-xs sm:text-sm font-semibold">Action</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {inventoryData.length === 0 ? (
                            <TableRow>
                              <TableCell colSpan={10} className="text-center py-8 text-gray-500">
                                No inventory data found. Add inventory data using the "Add Data" page to see records
                                here.
                              </TableCell>
                            </TableRow>
                          ) : (
                            inventoryData.map((item, index) => (
                              <TableRow key={item._id} className="hover:bg-muted/30">
                                <TableCell className="text-xs sm:text-sm font-medium">{index + 1}</TableCell>
                                <TableCell className="text-xs sm:text-sm">
                                  {new Date(item.date).toLocaleDateString()}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm">{item.branchName}</TableCell>
                                <TableCell className="text-xs sm:text-sm font-semibold text-green-600">
                                  +{item.kurczak}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-semibold text-green-600">
                                  +{item.wolowina}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-semibold text-green-600">
                                  +{item.kraftowe}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-semibold text-green-600">
                                  +{item.majonez}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-semibold text-green-600">
                                  +{item.ketchup}
                                </TableCell>
                                <TableCell className="text-xs sm:text-sm font-semibold text-green-600">
                                  +{item.gas || 0}
                                </TableCell>
                                <TableCell>
                                  <div className="flex flex-col sm:flex-row gap-1 sm:gap-2">
                                    <Button
                                      size="sm"
                                      variant="outline"
                                      className="h-7 px-2 text-xs bg-transparent"
                                      onClick={() => handleEdit(item)}
                                    >
                                      <Edit className="h-3 w-3 mr-1" />
                                      Edit
                                    </Button>
                                    <Button
                                      size="sm"
                                      variant="outline"
                                      className="h-7 px-2 text-xs bg-transparent"
                                      onClick={() => handleView(item)}
                                    >
                                      <Eye className="h-3 w-3 mr-1" />
                                      View
                                    </Button>
                                    <Button
                                      size="sm"
                                      variant="outline"
                                      className="h-7 px-2 text-xs text-red-600 hover:text-red-700 bg-transparent"
                                      onClick={() => {
                                        setSelectedItem(item)
                                        setIsDeleteModalOpen(true)
                                      }}
                                    >
                                      <Trash2 className="h-3 w-3 mr-1" />
                                      Delete
                                    </Button>
                                  </div>
                                </TableCell>
                              </TableRow>
                            ))
                          )}
                        </TableBody>
                      </Table>
                    </div>
                  </div>
                </TabsContent>
              </Tabs>
            )}
          </CardContent>
        </Card>

        <Dialog open={isViewModalOpen} onOpenChange={setIsViewModalOpen}>
          <DialogContent className="max-w-4xl">
            <DialogHeader>
              <DialogTitle>View Inventory Record</DialogTitle>
              <DialogDescription>Complete details of the selected inventory record.</DialogDescription>
            </DialogHeader>
            {selectedItem && (
              <div className="grid grid-cols-3 gap-4 py-4">
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Date</Label>
                  <p className="text-sm">{new Date(selectedItem.date).toLocaleDateString()}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Branch</Label>
                  <p className="text-sm">{selectedItem.branchName}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Type</Label>
                  <p className="text-sm capitalize">{selectedItem.type.replace("-", " ")}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Kurczak</Label>
                  <p className="text-sm font-semibold">{selectedItem.kurczak}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Wolowina</Label>
                  <p className="text-sm font-semibold">{selectedItem.wolowina}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Kraftowe</Label>
                  <p className="text-sm font-semibold">{selectedItem.kraftowe}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Majonez</Label>
                  <p className="text-sm font-semibold">{selectedItem.majonez}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Ketchup</Label>
                  <p className="text-sm font-semibold">{selectedItem.ketchup}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Gas</Label>
                  <p className="text-sm font-semibold">{selectedItem.gas}</p>
                </div>
                {selectedItem.notes && (
                  <div className="col-span-3 space-y-2">
                    <Label className="text-sm font-medium text-gray-600">Notes</Label>
                    <p className="text-sm">{selectedItem.notes}</p>
                  </div>
                )}
              </div>
            )}
          </DialogContent>
        </Dialog>

        <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
          <DialogContent className="max-w-4xl">
            <DialogHeader>
              <DialogTitle>Edit Inventory Record</DialogTitle>
              <DialogDescription>Update the inventory record details.</DialogDescription>
            </DialogHeader>
            <div className="grid grid-cols-3 gap-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="edit-date">Date</Label>
                <Input
                  id="edit-date"
                  type="date"
                  value={editFormData.date || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, date: e.target.value })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-branch">Branch</Label>
                <Select
                  value={editFormData.branchId || ""}
                  onValueChange={(value) => setEditFormData({ ...editFormData, branchId: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select branch" />
                  </SelectTrigger>
                  <SelectContent>
                    {branches?.map((branch) => (
                      <SelectItem key={branch._id} value={branch._id}>
                        {branch.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-type">Type</Label>
                <Select
                  value={editFormData.type || ""}
                  onValueChange={(value) => setEditFormData({ ...editFormData, type: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select type" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="daily-consumption">Daily Consumption</SelectItem>
                    <SelectItem value="stock-level">Stock Level</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-kurczak">Kurczak</Label>
                <Input
                  id="edit-kurczak"
                  type="number"
                  value={editFormData.kurczak || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, kurczak: Number(e.target.value) })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-wolowina">Wolowina</Label>
                <Input
                  id="edit-wolowina"
                  type="number"
                  value={editFormData.wolowina || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, wolowina: Number(e.target.value) })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-kraftowe">Kraftowe</Label>
                <Input
                  id="edit-kraftowe"
                  type="number"
                  value={editFormData.kraftowe || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, kraftowe: Number(e.target.value) })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-majonez">Majonez</Label>
                <Input
                  id="edit-majonez"
                  type="number"
                  value={editFormData.majonez || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, majonez: Number(e.target.value) })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-ketchup">Ketchup</Label>
                <Input
                  id="edit-ketchup"
                  type="number"
                  value={editFormData.ketchup || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, ketchup: Number(e.target.value) })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-gas">Gas</Label>
                <Input
                  id="edit-gas"
                  type="number"
                  value={editFormData.gas || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, gas: Number(e.target.value) })}
                />
              </div>
              <div className="col-span-3 space-y-2">
                <Label htmlFor="edit-notes">Notes</Label>
                <Input
                  id="edit-notes"
                  value={editFormData.notes || ""}
                  onChange={(e) => setEditFormData({ ...editFormData, notes: e.target.value })}
                  placeholder="Optional notes"
                />
              </div>
            </div>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsEditModalOpen(false)}>
                Cancel
              </Button>
              <Button onClick={handleSaveEdit}>Save Changes</Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <Dialog open={isDeleteModalOpen} onOpenChange={setIsDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Inventory Record</DialogTitle>
              <DialogDescription>
                Are you sure you want to delete this inventory record? This action cannot be undone.
              </DialogDescription>
            </DialogHeader>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsDeleteModalOpen(false)}>
                Cancel
              </Button>
              <Button variant="destructive" onClick={handleDelete}>
                Delete
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  )
}
