"use client"

import { useState, useEffect } from "react"
import { useSearchParams } from "next/navigation"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Download, FileText, Loader2, Calendar, MapPin } from "lucide-react"
import {
  Breadcrumb,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbList,
  BreadcrumbPage,
  BreadcrumbSeparator,
} from "@/components/ui/breadcrumb"
import DashboardLayout from "@/components/dashboard-layout"
import { useBranches } from "@/hooks/use-dashboard-data"
import { useSalesReport, useExpenseReport, useInventoryReport } from "@/hooks/use-reports-data"
import { useToast } from "@/hooks/use-toast"

export default function ExportReportsPage() {
  const searchParams = useSearchParams()
  const { toast } = useToast()

  const [filters, setFilters] = useState({
    branchId: "all",
    fromDate: new Date().toISOString().split("T")[0],
    toDate: new Date().toISOString().split("T")[0],
  })

  const { branches, isLoading: branchesLoading } = useBranches()
  const { data: salesData, isLoading: salesLoading, error: salesError } = useSalesReport(filters)
  const { data: expenseData, isLoading: expenseLoading, error: expenseError } = useExpenseReport(filters)
  const { data: inventoryData, isLoading: inventoryLoading, error: inventoryError } = useInventoryReport(filters)

  // Get parameters from URL
  useEffect(() => {
    const fromDate = searchParams.get("fromDate")
    const toDate = searchParams.get("toDate")
    const branchId = searchParams.get("branchId")

    if (fromDate || toDate || branchId) {
      setFilters({
        fromDate: fromDate || new Date().toISOString().split("T")[0],
        toDate: toDate || new Date().toISOString().split("T")[0],
        branchId: branchId || "all",
      })
    }
  }, [searchParams])

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("pl-PL", {
      style: "currency",
      currency: "PLN",
    }).format(amount)
  }

  const getSelectedBranchName = () => {
    if (filters.branchId === "all") return "All Branches"
    const branch = branches.find((b) => b._id === filters.branchId)
    return branch?.name || "Unknown Branch"
  }

  const handleExportPDF = () => {
    toast({
      title: "PDF Export Started",
      description: "Your PDF report is being generated...",
    })
    // PDF export logic would go here
    setTimeout(() => {
      toast({
        title: "PDF Export Complete",
        description: "Your PDF report has been downloaded successfully.",
      })
    }, 2000)
  }

  const handleExportExcel = () => {
    toast({
      title: "Excel Export Started",
      description: "Your Excel report is being generated...",
    })
    // Excel export logic would go here
    setTimeout(() => {
      toast({
        title: "Excel Export Complete",
        description: "Your Excel report has been downloaded successfully.",
      })
    }, 2000)
  }

  return (
    <DashboardLayout>
      <div className="p-3 sm:p-4 lg:p-6 xl:p-8">
        {/* Breadcrumb */}
        <div className="mb-4 sm:mb-6">
          <Breadcrumb>
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink href="/">Dashboard</BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbPage>Export Reports</BreadcrumbPage>
              </BreadcrumbItem>
            </BreadcrumbList>
          </Breadcrumb>
        </div>

        {/* Header */}
        <div className="mb-6 sm:mb-8">
          <div className="flex items-center gap-3 mb-2">
            <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-green-500 to-blue-600 flex items-center justify-center shadow-lg">
              <Download className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
            </div>
            <div>
              <h1 className="text-xl sm:text-2xl md:text-3xl font-bold bg-gradient-to-r from-green-600 to-blue-600 bg-clip-text text-transparent">
                Export Reports
              </h1>
              <p className="text-sm sm:text-base text-muted-foreground">
                Comprehensive data export in Excel-like format with PDF and Excel download options
              </p>
            </div>
          </div>
        </div>

        {/* Export Parameters Display */}
        <Card className="border-0 shadow-lg bg-gradient-to-r from-green-50 to-blue-50 backdrop-blur-sm ring-1 ring-green-200/50 mb-6">
          <CardContent className="px-3 sm:px-6 py-4">
            <div className="flex items-center gap-2 mb-3">
              <div className="h-6 w-6 rounded-full bg-green-500 flex items-center justify-center">
                <span className="text-white text-xs font-bold">✓</span>
              </div>
              <h3 className="text-sm font-semibold text-green-800">Export Parameters</h3>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
              <div className="flex items-center gap-3 p-3 bg-white/60 rounded-lg border border-green-200">
                <Calendar className="h-5 w-5 text-green-600" />
                <div>
                  <p className="text-xs font-medium text-gray-600">Date Range</p>
                  <p className="text-sm font-semibold text-gray-900">
                    {filters.fromDate} to {filters.toDate}
                  </p>
                </div>
              </div>
              <div className="flex items-center gap-3 p-3 bg-white/60 rounded-lg border border-green-200">
                <MapPin className="h-5 w-5 text-green-600" />
                <div>
                  <p className="text-xs font-medium text-gray-600">Branch</p>
                  <p className="text-sm font-semibold text-gray-900">{getSelectedBranchName()}</p>
                </div>
              </div>
              <div className="flex items-center gap-2">
                <Button onClick={handleExportPDF} className="flex-1 bg-red-600 hover:bg-red-700">
                  <FileText className="mr-2 h-4 w-4" />
                  Export PDF
                </Button>
                <Button onClick={handleExportExcel} className="flex-1 bg-green-600 hover:bg-green-700">
                  <Download className="mr-2 h-4 w-4" />
                  Export Excel
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Export Data Tabs */}
        <Tabs defaultValue="sales" className="space-y-4 sm:space-y-6">
          <div className="overflow-x-auto">
            <TabsList className="grid w-max grid-cols-4 gap-1 sm:gap-2 p-1">
              <TabsTrigger value="sales" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Sales Data
              </TabsTrigger>
              <TabsTrigger value="expenses" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Expenses Data
              </TabsTrigger>
              <TabsTrigger value="inventory" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                Inventory Data
              </TabsTrigger>
              <TabsTrigger value="if-any" className="text-xs sm:text-sm px-2 sm:px-4 py-2">
                IF ANY Data
              </TabsTrigger>
            </TabsList>
          </div>

          {/* Sales Data Tab */}
          <TabsContent value="sales" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-green-50/30">
              <CardHeader className="text-center pb-4 sm:pb-6">
                <CardTitle className="text-xl sm:text-2xl md:text-3xl font-bold">Diamond Kebab</CardTitle>
                <div className="space-y-1 sm:space-y-2 text-sm sm:text-base text-muted-foreground">
                  <p className="text-lg sm:text-xl font-semibold">{getSelectedBranchName()}</p>
                  <p className="text-base sm:text-lg font-medium">Sales Export Report</p>
                  <p>For the Period Of</p>
                  <p>
                    ({filters.fromDate} to {filters.toDate})
                  </p>
                </div>
              </CardHeader>
              <CardContent className="px-3 sm:px-6">
                {salesLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading sales data...
                  </div>
                ) : salesError ? (
                  <div className="text-center py-8 text-red-600">Error loading sales data: {salesError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-green-50">
                          <th className="border border-gray-300 px-2 py-2 text-left">Serial No</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Date</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Branch</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash in shop</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash delivery</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash wolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash bolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash pyszne</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash uber eats</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Cash glovo</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card in shop</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card delivery</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card wolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card bolt</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card pyszne</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card uber eats</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Card glovo</th>
                          <th className="border border-gray-300 px-2 py-2 text-left">Total</th>
                        </tr>
                      </thead>
                      <tbody>
                        {salesData.length === 0 ? (
                          <tr>
                            <td colSpan={18} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No sales data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          salesData.map((sale, index) => (
                            <tr key={sale._id}>
                              <td className="border border-gray-300 px-2 py-2">{index + 1}</td>
                              <td className="border border-gray-300 px-2 py-2">
                                {new Date(sale.date).toLocaleDateString()}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{sale.branchName}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashInShop)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashDelivery)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashWolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashBolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashPyszne)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashUberEats)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cashGlovo)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardInShop)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardDelivery)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardWolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardBolt)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardPyszne)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardUberEats)}</td>
                              <td className="border border-gray-300 px-2 py-2">{formatCurrency(sale.cardGlovo)}</td>
                              <td className="border border-gray-300 px-2 py-2 font-semibold">
                                {formatCurrency(sale.totalSales)}
                              </td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Expenses Data Tab */}
          <TabsContent value="expenses" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-red-50/30">
              <CardHeader className="text-center">
                <CardTitle className="text-xl sm:text-2xl font-bold">Diamond Kebab</CardTitle>
                <div className="space-y-1 text-sm sm:text-base text-muted-foreground">
                  <p className="text-lg font-semibold">{getSelectedBranchName()}</p>
                  <p className="text-base font-medium">Expenses Export Report</p>
                  <p>For the Period Of</p>
                  <p>
                    ({filters.fromDate} to {filters.toDate})
                  </p>
                </div>
              </CardHeader>
              <CardContent>
                {expenseLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading expense data...
                  </div>
                ) : expenseError ? (
                  <div className="text-center py-8 text-red-600">Error loading expense data: {expenseError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-red-50">
                          <th className="border border-gray-300 px-2 py-2">Serial No</th>
                          <th className="border border-gray-300 px-2 py-2">Date</th>
                          <th className="border border-gray-300 px-2 py-2">Branch</th>
                          <th className="border border-gray-300 px-2 py-2">Category</th>
                          <th className="border border-gray-300 px-2 py-2">Description</th>
                          <th className="border border-gray-300 px-2 py-2">Payment Method</th>
                          <th className="border border-gray-300 px-2 py-2">Amount</th>
                          <th className="border border-gray-300 px-2 py-2">Faktura No</th>
                          <th className="border border-gray-300 px-2 py-2">Notes</th>
                        </tr>
                      </thead>
                      <tbody>
                        {expenseData.length === 0 ? (
                          <tr>
                            <td colSpan={9} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No expense data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          expenseData.map((expense, index) => (
                            <tr key={expense._id}>
                              <td className="border border-gray-300 px-2 py-2">{index + 1}</td>
                              <td className="border border-gray-300 px-2 py-2">
                                {new Date(expense.date).toLocaleDateString()}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{expense.branchName}</td>
                              <td className="border border-gray-300 px-2 py-2 capitalize">
                                {expense.category.replace("-", " ")}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{expense.description}</td>
                              <td className="border border-gray-300 px-2 py-2 capitalize">{expense.paymentMethod}</td>
                              <td className="border border-gray-300 px-2 py-2 font-semibold">
                                {formatCurrency(expense.amount)}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{expense.fakturaNo || "-"}</td>
                              <td className="border border-gray-300 px-2 py-2">{expense.notes || "-"}</td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Inventory Data Tab */}
          <TabsContent value="inventory" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-blue-50/30">
              <CardHeader className="text-center">
                <CardTitle className="text-xl sm:text-2xl font-bold">Diamond Kebab</CardTitle>
                <div className="space-y-1 text-sm sm:text-base text-muted-foreground">
                  <p className="text-lg font-semibold">{getSelectedBranchName()}</p>
                  <p className="text-base font-medium">Inventory Export Report</p>
                  <p>For the Period Of</p>
                  <p>
                    ({filters.fromDate} to {filters.toDate})
                  </p>
                </div>
              </CardHeader>
              <CardContent>
                {inventoryLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading inventory data...
                  </div>
                ) : inventoryError ? (
                  <div className="text-center py-8 text-red-600">Error loading inventory data: {inventoryError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-blue-50">
                          <th className="border border-gray-300 px-2 py-2">Serial No</th>
                          <th className="border border-gray-300 px-2 py-2">Date</th>
                          <th className="border border-gray-300 px-2 py-2">Branch</th>
                          <th className="border border-gray-300 px-2 py-2">Type</th>
                          <th className="border border-gray-300 px-2 py-2">Kurczak (KG)</th>
                          <th className="border border-gray-300 px-2 py-2">Wolowina (KG)</th>
                          <th className="border border-gray-300 px-2 py-2">Kraftowe (KG)</th>
                          <th className="border border-gray-300 px-2 py-2">Majonez (KG)</th>
                          <th className="border border-gray-300 px-2 py-2">Ketchup (KG)</th>
                          <th className="border border-gray-300 px-2 py-2">Smabal (KG)</th>
                          <th className="border border-gray-300 px-2 py-2">Yogurt (KG)</th>
                          <th className="border border-gray-300 px-2 py-2">Pita 20cm (PC)</th>
                          <th className="border border-gray-300 px-2 py-2">Pita 30cm (PC)</th>
                          <th className="border border-gray-300 px-2 py-2">Gas (PC)</th>
                          <th className="border border-gray-300 px-2 py-2">Notes</th>
                        </tr>
                      </thead>
                      <tbody>
                        {inventoryData.length === 0 ? (
                          <tr>
                            <td colSpan={15} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No inventory data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          inventoryData.map((inventory, index) => (
                            <tr key={inventory._id}>
                              <td className="border border-gray-300 px-2 py-2">{index + 1}</td>
                              <td className="border border-gray-300 px-2 py-2">
                                {new Date(inventory.date).toLocaleDateString()}
                              </td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.branchName}</td>
                              <td className="border border-gray-300 px-2 py-2 capitalize">{inventory.type}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.kurczak}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.wolowina}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.kraftowe}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.majonez}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.ketchup}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.smabal}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.yogurt}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.pita_20_cm}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.pita_30_cm}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.gas}</td>
                              <td className="border border-gray-300 px-2 py-2">{inventory.notes || "-"}</td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* IF ANY Data Tab */}
          <TabsContent value="if-any" className="space-y-4 sm:space-y-6">
            <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-purple-50/30">
              <CardHeader className="text-center">
                <CardTitle className="text-xl sm:text-2xl font-bold">Diamond Kebab</CardTitle>
                <div className="space-y-1 text-sm sm:text-base text-muted-foreground">
                  <p className="text-lg font-semibold">{getSelectedBranchName()}</p>
                  <p className="text-base font-medium">IF ANY Export Report</p>
                  <p>For the Period Of</p>
                  <p>
                    ({filters.fromDate} to {filters.toDate})
                  </p>
                </div>
              </CardHeader>
              <CardContent>
                {expenseLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin mr-2" />
                    Loading IF ANY data...
                  </div>
                ) : expenseError ? (
                  <div className="text-center py-8 text-red-600">Error loading IF ANY data: {expenseError}</div>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full border-collapse border border-gray-300 text-xs sm:text-sm">
                      <thead>
                        <tr className="bg-purple-50">
                          <th className="border border-gray-300 px-2 py-2">Serial No</th>
                          <th className="border border-gray-300 px-2 py-2">Date</th>
                          <th className="border border-gray-300 px-2 py-2">Branch</th>
                          <th className="border border-gray-300 px-2 py-2">Description</th>
                          <th className="border border-gray-300 px-2 py-2">Amount</th>
                          <th className="border border-gray-300 px-2 py-2">Payment Date</th>
                          <th className="border border-gray-300 px-2 py-2">Payment Method</th>
                          <th className="border border-gray-300 px-2 py-2">Notes</th>
                        </tr>
                      </thead>
                      <tbody>
                        {expenseData.filter((expense) => expense.category === "if-any").length === 0 ? (
                          <tr>
                            <td colSpan={8} className="border border-gray-300 px-4 py-8 text-center text-gray-500">
                              No IF ANY data available for the selected period
                            </td>
                          </tr>
                        ) : (
                          expenseData
                            .filter((expense) => expense.category === "if-any")
                            .map((expense, index) => (
                              <tr key={expense._id}>
                                <td className="border border-gray-300 px-2 py-2">{index + 1}</td>
                                <td className="border border-gray-300 px-2 py-2">
                                  {new Date(expense.date).toLocaleDateString()}
                                </td>
                                <td className="border border-gray-300 px-2 py-2">{expense.branchName}</td>
                                <td className="border border-gray-300 px-2 py-2">{expense.description}</td>
                                <td className="border border-gray-300 px-2 py-2 font-semibold">
                                  {formatCurrency(expense.amount)}
                                </td>
                                <td className="border border-gray-300 px-2 py-2">
                                  {expense.paymentDate ? new Date(expense.paymentDate).toLocaleDateString() : "-"}
                                </td>
                                <td className="border border-gray-300 px-2 py-2 capitalize">{expense.paymentMethod}</td>
                                <td className="border border-gray-300 px-2 py-2">{expense.notes || "-"}</td>
                              </tr>
                            ))
                        )}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </DashboardLayout>
  )
}
