"use client"

import { useState, useEffect } from "react"
import DashboardLayout from "@/components/dashboard-layout"
import { Edit, Eye, Trash2, FileText, Loader2, Calendar, Building2, Filter, X } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { useExpenseReport } from "@/hooks/use-reports-data"
import { useBranches } from "@/hooks/use-dashboard-data"
import { useToast } from "@/hooks/use-toast"

export default function ExpenseReportPage() {
  const [activeTab, setActiveTab] = useState("managing-expenses")
  const [selectedExpense, setSelectedExpense] = useState(null)
  const [isViewModalOpen, setIsViewModalOpen] = useState(false)
  const [isEditModalOpen, setIsEditModalOpen] = useState(false)
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false)
  const [isAddModalOpen, setIsAddModalOpen] = useState(false)
  const { toast } = useToast()

  const [filters, setFilters] = useState({
    fromDate: "",
    toDate: "",
    branchId: "all",
    category: "managing-expenses",
  })

  const { branches } = useBranches()

  const {
    data: expenseData,
    isLoading,
    error,
    mutate: refreshExpenses,
  } = useExpenseReport({
    category: activeTab === "all" ? undefined : activeTab,
    branchId: filters.branchId === "all" ? undefined : filters.branchId,
    fromDate: filters.fromDate || undefined,
    toDate: filters.toDate || undefined,
  })

  useEffect(() => {
    setFilters((prev) => ({
      ...prev,
      category: activeTab,
    }))
  }, [activeTab])

  const handleFilterChange = (key: string, value: string) => {
    setFilters((prev) => ({
      ...prev,
      [key]: value,
    }))
  }

  const clearFilters = () => {
    setFilters({
      fromDate: "",
      toDate: "",
      branchId: "all",
      category: activeTab,
    })
  }

  const handleView = (expense) => {
    setSelectedExpense(expense)
    setIsViewModalOpen(true)
  }

  const handleEdit = (expense) => {
    setSelectedExpense(expense)
    setIsEditModalOpen(true)
  }

  const handleDelete = async (expense) => {
    try {
      const response = await fetch(`/api/expenses/${expense._id}`, {
        method: "DELETE",
      })

      if (response.ok) {
        toast({
          title: "Success",
          description: "Expense deleted successfully",
          variant: "default",
        })
        refreshExpenses()
        setIsDeleteModalOpen(false)
      } else {
        throw new Error("Failed to delete expense")
      }
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to delete expense",
        variant: "destructive",
      })
    }
  }

  const handleSaveEdit = async (formData) => {
    try {
      const response = await fetch(`/api/expenses/${selectedExpense._id}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(formData),
      })

      if (response.ok) {
        toast({
          title: "Success",
          description: "Expense updated successfully",
          variant: "default",
        })
        refreshExpenses()
        setIsEditModalOpen(false)
      } else {
        throw new Error("Failed to update expense")
      }
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update expense",
        variant: "destructive",
      })
    }
  }

  const expenseTabs = [
    { id: "all", label: "All Expenses", icon: "📊" },
    { id: "managing-expenses", label: "Managing Expenses", icon: "📊" },
    { id: "salaries", label: "Salaries", icon: "👥" },
    { id: "fooding-expense", label: "Fooding Expense", icon: "🍽️" },
    { id: "salaries-calculations", label: "Salaries Calculations", icon: "🧮" },
    { id: "cost-of-sales", label: "Cost Of Sales", icon: "💰" },
    { id: "utilities", label: "Utilities", icon: "⚡" },
    { id: "office-expenses", label: "Office Expenses", icon: "🏢" },
    { id: "delivery-cost", label: "Delivery Cost", icon: "🚚" },
    { id: "shop-rent", label: "Shop Rent", icon: "🏪" },
    { id: "home-expenses", label: "Home Expenses", icon: "🏠" },
    { id: "repair-maintenance", label: "Repair & Maintenance", icon: "🔧" },
    { id: "bank-charges", label: "Bank Charges", icon: "🏦" },
    { id: "others", label: "Others", icon: "📋" },
    { id: "personal-withdraw", label: "Personal Withdraw", icon: "💳" },
  ]

  const renderExpenseTable = () => {
    if (isLoading) {
      return (
        <div className="flex items-center justify-center py-8">
          <Loader2 className="h-6 w-6 animate-spin mr-2" />
          Loading expense data from database...
        </div>
      )
    }

    if (error) {
      return <div className="text-center py-8 text-red-600">Error loading expense data: {error}</div>
    }

    const isSalaryCategory = activeTab === "salaries" || activeTab === "salaries-calculations"

    return (
      <Table>
        <TableHeader>
          <TableRow>
            <TableHead>Serial No</TableHead>
            <TableHead>Date</TableHead>
            <TableHead>Branch</TableHead>
            <TableHead>Category</TableHead>
            {isSalaryCategory && <TableHead>Employee</TableHead>}
            {isSalaryCategory && <TableHead>Hours</TableHead>}
            {isSalaryCategory && <TableHead>Rate</TableHead>}
            <TableHead>Description</TableHead>
            <TableHead>Amount</TableHead>
            <TableHead>Payment Method</TableHead>
            <TableHead>Faktura No.</TableHead>
            <TableHead>Actions</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {expenseData.length === 0 ? (
            <TableRow>
              <TableCell colSpan={isSalaryCategory ? 12 : 9} className="text-center py-8 text-gray-500">
                No expense data found for the selected filters. Add expense data using the "Add Data" page to see
                records here.
              </TableCell>
            </TableRow>
          ) : (
            expenseData.map((expense, index) => (
              <TableRow key={expense._id}>
                <TableCell>{index + 1}</TableCell>
                <TableCell>{new Date(expense.date).toLocaleDateString()}</TableCell>
                <TableCell>{expense.branchName}</TableCell>
                <TableCell>
                  <span className="px-2 py-1 rounded-full text-xs bg-blue-100 text-blue-800">{expense.category}</span>
                </TableCell>
                {isSalaryCategory && (
                  <>
                    <TableCell>{expense.employeeName || "N/A"}</TableCell>
                    <TableCell>{expense.totalHours || "N/A"}</TableCell>
                    <TableCell>{expense.hourlyRate ? `$${expense.hourlyRate}` : "N/A"}</TableCell>
                  </>
                )}
                <TableCell>{expense.description}</TableCell>
                <TableCell className="font-semibold text-green-600">${expense.amount.toLocaleString()}</TableCell>
                <TableCell className="capitalize">{expense.paymentMethod}</TableCell>
                <TableCell>{expense.fakturaNo || "-"}</TableCell>
                <TableCell>
                  <div className="flex gap-1">
                    <Button
                      size="sm"
                      variant="outline"
                      className="h-7 px-2 bg-transparent"
                      onClick={() => handleView(expense)}
                    >
                      <Eye className="h-3 w-3" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      className="h-7 px-2 bg-transparent"
                      onClick={() => handleEdit(expense)}
                    >
                      <Edit className="h-3 w-3" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      className="h-7 px-2 bg-transparent text-red-600 hover:text-red-700"
                      onClick={() => {
                        setSelectedExpense(expense)
                        setIsDeleteModalOpen(true)
                      }}
                    >
                      <Trash2 className="h-3 w-3" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))
          )}
        </TableBody>
      </Table>
    )
  }

  return (
    <DashboardLayout user={{ name: "John Doe", email: "john@example.com" }}>
      <div className="p-3 sm:p-4 lg:p-6">
        {/* Header */}
        <div className="mb-6 sm:mb-8">
          <div className="flex items-center gap-3 mb-2">
            <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-red-500 to-orange-600 flex items-center justify-center shadow-lg">
              <FileText className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
            </div>
            <div>
              <h1 className="text-xl sm:text-2xl lg:text-3xl font-bold bg-gradient-to-r from-red-600 to-orange-600 bg-clip-text text-transparent">
                Expense Reports
              </h1>
              <p className="text-sm sm:text-base text-muted-foreground">
                Comprehensive expense tracking and management ({expenseData.length} records from database)
              </p>
            </div>
          </div>
        </div>

        <Card className="mb-6 border-0 shadow-lg bg-gradient-to-br from-blue-50 to-indigo-50/50">
          <CardHeader className="pb-4">
            <CardTitle className="flex items-center gap-2 text-lg text-blue-700">
              <Filter className="h-5 w-5" />
              Filters
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
              <div className="space-y-2">
                <Label htmlFor="fromDate" className="text-sm font-medium">
                  From Date
                </Label>
                <div className="relative">
                  <Calendar className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    id="fromDate"
                    type="date"
                    value={filters.fromDate}
                    onChange={(e) => handleFilterChange("fromDate", e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="toDate" className="text-sm font-medium">
                  To Date
                </Label>
                <div className="relative">
                  <Calendar className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    id="toDate"
                    type="date"
                    value={filters.toDate}
                    onChange={(e) => handleFilterChange("toDate", e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="branch" className="text-sm font-medium">
                  Branch
                </Label>
                <div className="relative">
                  <Building2 className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400 z-10" />
                  <Select value={filters.branchId} onValueChange={(value) => handleFilterChange("branchId", value)}>
                    <SelectTrigger className="pl-10">
                      <SelectValue placeholder="Select branch" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Branches</SelectItem>
                      {branches.map((branch) => (
                        <SelectItem key={branch._id} value={branch._id}>
                          {branch.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </div>
              <div className="flex items-end">
                <Button variant="outline" onClick={clearFilters} className="w-full bg-transparent">
                  <X className="h-4 w-4 mr-2" />
                  Clear Filters
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Expense Categories Tabs */}
        <Card className="border-0 shadow-lg bg-white/80 backdrop-blur-sm">
          <CardHeader className="pb-4">
            <div className="flex items-center justify-between">
              <CardTitle className="text-lg sm:text-xl">
                Expense Categories -{" "}
                {activeTab === "all" ? "All Categories" : expenseTabs.find((tab) => tab.id === activeTab)?.label}
              </CardTitle>
            </div>
          </CardHeader>
          <CardContent>
            <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
              <div className="overflow-x-auto">
                <TabsList className="inline-flex h-auto p-1 bg-muted/50 min-w-max">
                  {expenseTabs.map((tab) => (
                    <TabsTrigger
                      key={tab.id}
                      value={tab.id}
                      className="whitespace-nowrap text-xs sm:text-sm px-2 sm:px-3 py-2 data-[state=active]:bg-white data-[state=active]:shadow-sm"
                    >
                      <span className="mr-1 sm:mr-2">{tab.icon}</span>
                      {tab.label}
                    </TabsTrigger>
                  ))}
                </TabsList>
              </div>

              {expenseTabs.map((tab) => (
                <TabsContent key={tab.id} value={tab.id} className="space-y-4">
                  <div className="rounded-lg border bg-white overflow-hidden">
                    <div className="overflow-x-auto">{renderExpenseTable()}</div>
                  </div>
                </TabsContent>
              ))}
            </Tabs>
          </CardContent>
        </Card>

        <Dialog open={isViewModalOpen} onOpenChange={setIsViewModalOpen}>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>View Expense Details</DialogTitle>
              <DialogDescription>Complete details of the selected expense record.</DialogDescription>
            </DialogHeader>
            {selectedExpense && (
              <div className="grid grid-cols-2 gap-4 py-4">
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Date</Label>
                  <p className="text-sm">{new Date(selectedExpense.date).toLocaleDateString()}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Branch</Label>
                  <p className="text-sm">{selectedExpense.branchName}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Category</Label>
                  <p className="text-sm">{selectedExpense.category}</p>
                </div>
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Amount</Label>
                  <p className="text-sm font-semibold text-green-600">${selectedExpense.amount.toLocaleString()}</p>
                </div>
                {selectedExpense.employeeName && (
                  <div className="space-y-2">
                    <Label className="text-sm font-medium text-gray-600">Employee</Label>
                    <p className="text-sm">{selectedExpense.employeeName}</p>
                  </div>
                )}
                <div className="space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Payment Method</Label>
                  <p className="text-sm capitalize">{selectedExpense.paymentMethod}</p>
                </div>
                <div className="col-span-2 space-y-2">
                  <Label className="text-sm font-medium text-gray-600">Description</Label>
                  <p className="text-sm">{selectedExpense.description}</p>
                </div>
              </div>
            )}
          </DialogContent>
        </Dialog>

        <Dialog open={isDeleteModalOpen} onOpenChange={setIsDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Expense</DialogTitle>
              <DialogDescription>
                Are you sure you want to delete this expense record? This action cannot be undone.
              </DialogDescription>
            </DialogHeader>
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsDeleteModalOpen(false)}>
                Cancel
              </Button>
              <Button variant="destructive" onClick={() => handleDelete(selectedExpense)}>
                Delete
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
          <DialogContent className="max-w-3xl">
            <DialogHeader>
              <DialogTitle>Edit Expense Record</DialogTitle>
              <DialogDescription>Update the expense record details.</DialogDescription>
            </DialogHeader>
            {selectedExpense && (
              <div className="grid grid-cols-2 gap-4 py-4">
                <div className="space-y-2">
                  <Label htmlFor="edit-date">Date</Label>
                  <Input
                    id="edit-date"
                    type="date"
                    defaultValue={new Date(selectedExpense.date).toISOString().split("T")[0]}
                    onChange={(e) => setSelectedExpense({ ...selectedExpense, date: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit-branch">Branch</Label>
                  <Select
                    defaultValue={selectedExpense.branchId}
                    onValueChange={(value) => setSelectedExpense({ ...selectedExpense, branchId: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select branch" />
                    </SelectTrigger>
                    <SelectContent>
                      {branches.map((branch) => (
                        <SelectItem key={branch._id} value={branch._id}>
                          {branch.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit-category">Category</Label>
                  <Select
                    defaultValue={selectedExpense.category}
                    onValueChange={(value) => setSelectedExpense({ ...selectedExpense, category: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select category" />
                    </SelectTrigger>
                    <SelectContent>
                      {expenseTabs
                        .filter((tab) => tab.id !== "all")
                        .map((tab) => (
                          <SelectItem key={tab.id} value={tab.id}>
                            {tab.label}
                          </SelectItem>
                        ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit-amount">Amount</Label>
                  <Input
                    id="edit-amount"
                    type="number"
                    defaultValue={selectedExpense.amount}
                    onChange={(e) => setSelectedExpense({ ...selectedExpense, amount: Number(e.target.value) })}
                  />
                </div>
                {(selectedExpense.category === "salaries" || selectedExpense.category === "salaries-calculations") && (
                  <>
                    <div className="space-y-2">
                      <Label htmlFor="edit-employee">Employee Name</Label>
                      <Input
                        id="edit-employee"
                        defaultValue={selectedExpense.employeeName || ""}
                        onChange={(e) => setSelectedExpense({ ...selectedExpense, employeeName: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="edit-hours">Total Hours</Label>
                      <Input
                        id="edit-hours"
                        type="number"
                        defaultValue={selectedExpense.totalHours || ""}
                        onChange={(e) => setSelectedExpense({ ...selectedExpense, totalHours: Number(e.target.value) })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="edit-rate">Hourly Rate</Label>
                      <Input
                        id="edit-rate"
                        type="number"
                        defaultValue={selectedExpense.hourlyRate || ""}
                        onChange={(e) => setSelectedExpense({ ...selectedExpense, hourlyRate: Number(e.target.value) })}
                      />
                    </div>
                  </>
                )}
                <div className="space-y-2">
                  <Label htmlFor="edit-payment">Payment Method</Label>
                  <Select
                    defaultValue={selectedExpense.paymentMethod}
                    onValueChange={(value) => setSelectedExpense({ ...selectedExpense, paymentMethod: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select payment method" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="cash">Cash</SelectItem>
                      <SelectItem value="card">Card</SelectItem>
                      <SelectItem value="bank-transfer">Bank Transfer</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="edit-faktura">Faktura No.</Label>
                  <Input
                    id="edit-faktura"
                    defaultValue={selectedExpense.fakturaNo || ""}
                    onChange={(e) => setSelectedExpense({ ...selectedExpense, fakturaNo: e.target.value })}
                    placeholder="Optional"
                  />
                </div>
                <div className="col-span-2 space-y-2">
                  <Label htmlFor="edit-description">Description</Label>
                  <Input
                    id="edit-description"
                    defaultValue={selectedExpense.description}
                    onChange={(e) => setSelectedExpense({ ...selectedExpense, description: e.target.value })}
                  />
                </div>
              </div>
            )}
            <DialogFooter>
              <Button variant="outline" onClick={() => setIsEditModalOpen(false)}>
                Cancel
              </Button>
              <Button onClick={() => handleSaveEdit(selectedExpense)}>Save Changes</Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  )
}
