"use client"

import { useState, useEffect } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Trash2, Edit, Search, UserPlus, Loader2, Eye, Building } from "lucide-react"
import { useToast } from "@/hooks/use-toast"
import DashboardLayout from "@/components/dashboard-layout"

interface Employee {
  _id: string
  name: string
  email?: string
  phone?: string
  position: string
  branchId: string
  hourlyRate?: number
  isActive: boolean
  startDate: string
  endDate?: string
  createdAt: string
  updatedAt: string
}

interface Branch {
  _id: string
  name: string
  address: string
  isActive: boolean
}

interface CreateEmployeeData {
  name: string
  email: string
  phone: string
  position: string
  branchId: string
  hourlyRate: number
  startDate: string
}

export default function EmployeesPage() {
  const [employees, setEmployees] = useState<Employee[]>([])
  const [branches, setBranches] = useState<Branch[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [isAddModalOpen, setIsAddModalOpen] = useState(false)
  const [isEditModalOpen, setIsEditModalOpen] = useState(false)
  const [isViewModalOpen, setIsViewModalOpen] = useState(false)
  const [selectedEmployee, setSelectedEmployee] = useState<Employee | null>(null)
  const [searchTerm, setSearchTerm] = useState("")
  const [branchFilter, setBranchFilter] = useState("all")
  const [statusFilter, setStatusFilter] = useState("all")

  const [newEmployee, setNewEmployee] = useState<CreateEmployeeData>({
    name: "",
    email: "",
    phone: "",
    position: "",
    branchId: "",
    hourlyRate: 0,
    startDate: new Date().toISOString().split("T")[0],
  })

  const { toast } = useToast()

  useEffect(() => {
    fetchEmployees()
    fetchBranches()
  }, [])

  const fetchEmployees = async () => {
    try {
      console.log("[v0] Fetching employees from database...")
      const response = await fetch("/api/employees")
      const result = await response.json()
      if (result.success) {
        setEmployees(result.data)
        console.log("[v0] Successfully loaded employees:", result.data.length)
      } else {
        setError(result.error || "Failed to fetch employees")
        console.error("[v0] Error fetching employees:", result.error)
      }
    } catch (error) {
      console.error("[v0] Network error fetching employees:", error)
      setError("Network error occurred")
    } finally {
      setLoading(false)
    }
  }

  const fetchBranches = async () => {
    try {
      console.log("[v0] Fetching branches for employee assignment...")
      const response = await fetch("/api/branches")
      const result = await response.json()
      if (result.success) {
        setBranches(result.data)
        console.log("[v0] Successfully loaded branches for assignment:", result.data.length)
      }
    } catch (error) {
      console.error("[v0] Error fetching branches:", error)
    }
  }

  const handleAddEmployee = async () => {
    try {
      console.log("[v0] Adding new employee to database...")
      const response = await fetch("/api/employees", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(newEmployee),
      })

      const result = await response.json()
      if (result.success) {
        setEmployees([result.data, ...employees])
        setIsAddModalOpen(false)
        setNewEmployee({
          name: "",
          email: "",
          phone: "",
          position: "",
          branchId: "",
          hourlyRate: 0,
          startDate: new Date().toISOString().split("T")[0],
        })
        toast({
          title: "Success",
          description: "Employee added successfully!",
          variant: "default",
        })
        console.log("[v0] Employee added successfully")
      } else {
        toast({
          title: "Error",
          description: "Failed to add employee: " + result.error,
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error adding employee:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  const handleEditEmployee = async () => {
    if (!selectedEmployee) return

    try {
      console.log("[v0] Updating employee in database...")
      const response = await fetch(`/api/employees/${selectedEmployee._id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(selectedEmployee),
      })

      const result = await response.json()
      if (result.success) {
        setEmployees(employees.map((emp) => (emp._id === selectedEmployee._id ? result.data : emp)))
        setIsEditModalOpen(false)
        setSelectedEmployee(null)
        toast({
          title: "Success",
          description: "Employee updated successfully!",
          variant: "default",
        })
        console.log("[v0] Employee updated successfully")
      } else {
        toast({
          title: "Error",
          description: "Failed to update employee: " + result.error,
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error updating employee:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  const handleDeleteEmployee = async (employeeId: string) => {
    if (!confirm("Are you sure you want to delete this employee?")) return

    try {
      console.log("[v0] Deleting employee from database...")
      const response = await fetch(`/api/employees/${employeeId}`, {
        method: "DELETE",
      })
      const result = await response.json()

      if (result.success) {
        setEmployees(employees.filter((employee) => employee._id !== employeeId))
        toast({
          title: "Success",
          description: "Employee deleted successfully!",
          variant: "default",
        })
        console.log("[v0] Employee deleted successfully")
      } else {
        toast({
          title: "Error",
          description: "Failed to delete employee: " + result.error,
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error deleting employee:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  const openEditModal = (employee: Employee) => {
    setSelectedEmployee(employee)
    setIsEditModalOpen(true)
  }

  const openViewModal = (employee: Employee) => {
    setSelectedEmployee(employee)
    setIsViewModalOpen(true)
  }

  const filteredEmployees = employees.filter((employee) => {
    const matchesSearch =
      employee.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      employee.position.toLowerCase().includes(searchTerm.toLowerCase()) ||
      (employee.email && employee.email.toLowerCase().includes(searchTerm.toLowerCase()))
    const matchesBranch = branchFilter === "all" || employee.branchId === branchFilter
    const matchesStatus =
      statusFilter === "all" ||
      (statusFilter === "active" && employee.isActive) ||
      (statusFilter === "inactive" && !employee.isActive)
    return matchesSearch && matchesBranch && matchesStatus
  })

  const getBranchName = (branchId: string) => {
    const branch = branches.find((b) => b._id === branchId)
    return branch?.name || "Unknown Branch"
  }

  if (loading) {
    return (
      <DashboardLayout>
        <div className="p-6">
          <div className="flex items-center justify-center h-64">
            <Loader2 className="h-6 w-6 animate-spin mr-2" />
            <div className="text-gray-600">Loading employees from database...</div>
          </div>
        </div>
      </DashboardLayout>
    )
  }

  if (error) {
    return (
      <DashboardLayout>
        <div className="p-6">
          <div className="text-center py-8 text-red-600">Error loading employees: {error}</div>
        </div>
      </DashboardLayout>
    )
  }

  return (
    <DashboardLayout>
      <div className="p-6 space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Employee Management</h1>
            <p className="text-gray-600">Manage employees and their information ({employees.length} total employees)</p>
          </div>
          <Dialog open={isAddModalOpen} onOpenChange={setIsAddModalOpen}>
            <DialogTrigger asChild>
              <Button className="bg-blue-600 hover:bg-blue-700">
                <UserPlus className="h-4 w-4 mr-2" />
                Add Employee
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-md">
              <DialogHeader>
                <DialogTitle>Add New Employee</DialogTitle>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="name">Full Name *</Label>
                  <Input
                    id="name"
                    value={newEmployee.name}
                    onChange={(e) => setNewEmployee({ ...newEmployee, name: e.target.value })}
                    placeholder="Enter full name"
                  />
                </div>
                <div>
                  <Label htmlFor="email">Email</Label>
                  <Input
                    id="email"
                    type="email"
                    value={newEmployee.email}
                    onChange={(e) => setNewEmployee({ ...newEmployee, email: e.target.value })}
                    placeholder="Enter email address"
                  />
                </div>
                <div>
                  <Label htmlFor="phone">Phone</Label>
                  <Input
                    id="phone"
                    value={newEmployee.phone}
                    onChange={(e) => setNewEmployee({ ...newEmployee, phone: e.target.value })}
                    placeholder="Enter phone number"
                  />
                </div>
                <div>
                  <Label htmlFor="position">Position *</Label>
                  <Input
                    id="position"
                    value={newEmployee.position}
                    onChange={(e) => setNewEmployee({ ...newEmployee, position: e.target.value })}
                    placeholder="Enter position"
                  />
                </div>
                <div>
                  <Label htmlFor="branch">Branch *</Label>
                  <Select
                    value={newEmployee.branchId}
                    onValueChange={(value) => setNewEmployee({ ...newEmployee, branchId: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select branch" />
                    </SelectTrigger>
                    <SelectContent>
                      {branches.map((branch) => (
                        <SelectItem key={branch._id} value={branch._id}>
                          {branch.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="hourlyRate">Hourly Rate</Label>
                  <Input
                    id="hourlyRate"
                    type="number"
                    value={newEmployee.hourlyRate}
                    onChange={(e) => setNewEmployee({ ...newEmployee, hourlyRate: Number(e.target.value) })}
                    placeholder="Enter hourly rate"
                  />
                </div>
                <div>
                  <Label htmlFor="startDate">Start Date</Label>
                  <Input
                    id="startDate"
                    type="date"
                    value={newEmployee.startDate}
                    onChange={(e) => setNewEmployee({ ...newEmployee, startDate: e.target.value })}
                  />
                </div>
                <Button onClick={handleAddEmployee} className="w-full">
                  Add Employee
                </Button>
              </div>
            </DialogContent>
          </Dialog>
        </div>

        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <CardTitle>All Employees</CardTitle>
              <div className="flex items-center space-x-2">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    placeholder="Search employees..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10 w-64"
                  />
                </div>
                <Select value={branchFilter} onValueChange={setBranchFilter}>
                  <SelectTrigger className="w-40">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Branches</SelectItem>
                    {branches.map((branch) => (
                      <SelectItem key={branch._id} value={branch._id}>
                        {branch.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Select value={statusFilter} onValueChange={setStatusFilter}>
                  <SelectTrigger className="w-32">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Status</SelectItem>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="inactive">Inactive</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b">
                    <th className="text-left py-3 px-4">Name</th>
                    <th className="text-left py-3 px-4">Position</th>
                    <th className="text-left py-3 px-4">Branch</th>
                    <th className="text-left py-3 px-4">Email</th>
                    <th className="text-left py-3 px-4">Phone</th>
                    <th className="text-left py-3 px-4">Hourly Rate</th>
                    <th className="text-left py-3 px-4">Status</th>
                    <th className="text-left py-3 px-4">Start Date</th>
                    <th className="text-left py-3 px-4">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {filteredEmployees.length === 0 ? (
                    <tr>
                      <td colSpan={9} className="text-center py-8 text-gray-500">
                        {employees.length === 0
                          ? "No employees found in database"
                          : "No employees found matching your criteria"}
                      </td>
                    </tr>
                  ) : (
                    filteredEmployees.map((employee) => (
                      <tr key={employee._id} className="border-b hover:bg-gray-50">
                        <td className="py-3 px-4 font-medium">{employee.name}</td>
                        <td className="py-3 px-4 text-gray-600">{employee.position}</td>
                        <td className="py-3 px-4 text-gray-600">
                          <div className="flex items-center">
                            <Building className="h-4 w-4 mr-1 text-gray-400" />
                            {getBranchName(employee.branchId)}
                          </div>
                        </td>
                        <td className="py-3 px-4 text-gray-600">{employee.email || "-"}</td>
                        <td className="py-3 px-4 text-gray-600">{employee.phone || "-"}</td>
                        <td className="py-3 px-4 text-gray-600">
                          {employee.hourlyRate ? `$${employee.hourlyRate}/hr` : "-"}
                        </td>
                        <td className="py-3 px-4">
                          <Badge
                            className={employee.isActive ? "bg-green-100 text-green-800" : "bg-red-100 text-red-800"}
                          >
                            {employee.isActive ? "Active" : "Inactive"}
                          </Badge>
                        </td>
                        <td className="py-3 px-4 text-sm text-gray-600">
                          {new Date(employee.startDate).toLocaleDateString()}
                        </td>
                        <td className="py-3 px-4">
                          <div className="flex items-center space-x-2">
                            <Button variant="outline" size="sm" onClick={() => openViewModal(employee)}>
                              <Eye className="h-4 w-4" />
                            </Button>
                            <Button variant="outline" size="sm" onClick={() => openEditModal(employee)}>
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="outline"
                              size="sm"
                              className="text-red-600 hover:text-red-700 bg-transparent"
                              onClick={() => handleDeleteEmployee(employee._id)}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>

        {/* View Employee Modal */}
        <Dialog open={isViewModalOpen} onOpenChange={setIsViewModalOpen}>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>Employee Details</DialogTitle>
            </DialogHeader>
            {selectedEmployee && (
              <div className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Name</Label>
                    <p className="text-sm font-medium">{selectedEmployee.name}</p>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Position</Label>
                    <p className="text-sm">{selectedEmployee.position}</p>
                  </div>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Email</Label>
                    <p className="text-sm">{selectedEmployee.email || "-"}</p>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Phone</Label>
                    <p className="text-sm">{selectedEmployee.phone || "-"}</p>
                  </div>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Branch</Label>
                    <p className="text-sm">{getBranchName(selectedEmployee.branchId)}</p>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Hourly Rate</Label>
                    <p className="text-sm">
                      {selectedEmployee.hourlyRate ? `$${selectedEmployee.hourlyRate}/hr` : "-"}
                    </p>
                  </div>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Status</Label>
                    <Badge
                      className={selectedEmployee.isActive ? "bg-green-100 text-green-800" : "bg-red-100 text-red-800"}
                    >
                      {selectedEmployee.isActive ? "Active" : "Inactive"}
                    </Badge>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Start Date</Label>
                    <p className="text-sm">{new Date(selectedEmployee.startDate).toLocaleDateString()}</p>
                  </div>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Created</Label>
                  <p className="text-sm">{new Date(selectedEmployee.createdAt).toLocaleDateString()}</p>
                </div>
              </div>
            )}
          </DialogContent>
        </Dialog>

        {/* Edit Employee Modal */}
        <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>Edit Employee</DialogTitle>
            </DialogHeader>
            {selectedEmployee && (
              <div className="space-y-4">
                <div>
                  <Label htmlFor="editName">Full Name *</Label>
                  <Input
                    id="editName"
                    value={selectedEmployee.name}
                    onChange={(e) => setSelectedEmployee({ ...selectedEmployee, name: e.target.value })}
                    placeholder="Enter full name"
                  />
                </div>
                <div>
                  <Label htmlFor="editEmail">Email</Label>
                  <Input
                    id="editEmail"
                    type="email"
                    value={selectedEmployee.email || ""}
                    onChange={(e) => setSelectedEmployee({ ...selectedEmployee, email: e.target.value })}
                    placeholder="Enter email address"
                  />
                </div>
                <div>
                  <Label htmlFor="editPhone">Phone</Label>
                  <Input
                    id="editPhone"
                    value={selectedEmployee.phone || ""}
                    onChange={(e) => setSelectedEmployee({ ...selectedEmployee, phone: e.target.value })}
                    placeholder="Enter phone number"
                  />
                </div>
                <div>
                  <Label htmlFor="editPosition">Position *</Label>
                  <Input
                    id="editPosition"
                    value={selectedEmployee.position}
                    onChange={(e) => setSelectedEmployee({ ...selectedEmployee, position: e.target.value })}
                    placeholder="Enter position"
                  />
                </div>
                <div>
                  <Label htmlFor="editBranch">Branch *</Label>
                  <Select
                    value={selectedEmployee.branchId}
                    onValueChange={(value) => setSelectedEmployee({ ...selectedEmployee, branchId: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select branch" />
                    </SelectTrigger>
                    <SelectContent>
                      {branches.map((branch) => (
                        <SelectItem key={branch._id} value={branch._id}>
                          {branch.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="editHourlyRate">Hourly Rate</Label>
                  <Input
                    id="editHourlyRate"
                    type="number"
                    value={selectedEmployee.hourlyRate || 0}
                    onChange={(e) => setSelectedEmployee({ ...selectedEmployee, hourlyRate: Number(e.target.value) })}
                    placeholder="Enter hourly rate"
                  />
                </div>
                <div>
                  <Label htmlFor="editStatus">Status</Label>
                  <Select
                    value={selectedEmployee.isActive ? "active" : "inactive"}
                    onValueChange={(value) =>
                      setSelectedEmployee({ ...selectedEmployee, isActive: value === "active" })
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="active">Active</SelectItem>
                      <SelectItem value="inactive">Inactive</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <Button onClick={handleEditEmployee} className="w-full">
                  Update Employee
                </Button>
              </div>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  )
}
