"use client"
import { useState, useEffect } from "react"
import type React from "react"

import Link from "next/link"
import { Building, Edit, Trash2, FileBarChart, BarChart3, Search, Loader2, Plus } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  Breadcrumb,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbList,
  BreadcrumbPage,
  BreadcrumbSeparator,
} from "@/components/ui/breadcrumb"
import DashboardLayout from "@/components/dashboard-layout"
import { useToast } from "@/hooks/use-toast"

interface Branch {
  _id: string
  name: string
  address: string
  country?: string
  location?: string
  isActive: boolean
  createdAt: string
  updatedAt: string
}

interface Owner {
  _id: string
  name: string
  email: string
}

export default function BranchesPage() {
  const [branches, setBranches] = useState<Branch[]>([])
  const [owners, setOwners] = useState<Owner[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [createDialogOpen, setCreateDialogOpen] = useState(false)
  const [createLoading, setCreateLoading] = useState(false)
  const { toast } = useToast()

  const [newBranch, setNewBranch] = useState({
    name: "",
    address: "",
    country: "",
    location: "",
    ownerId: "",
  })

  useEffect(() => {
    fetchBranches()
    fetchOwners()
  }, [])

  const fetchBranches = async () => {
    try {
      console.log("[v0] Fetching branches from database...")
      const response = await fetch("/api/branches")
      const result = await response.json()

      if (result.success) {
        setBranches(result.data)
        console.log("[v0] Successfully loaded branches:", result.data.length)
      } else {
        setError(result.error || "Failed to fetch branches")
        console.error("[v0] Error fetching branches:", result.error)
      }
    } catch (error) {
      console.error("[v0] Network error fetching branches:", error)
      setError("Network error occurred")
    } finally {
      setLoading(false)
    }
  }

  const fetchOwners = async () => {
    try {
      const response = await fetch("/api/users")
      const result = await response.json()
      if (result.success) {
        // Filter users with owner role
        const ownerUsers = result.data.filter((user: any) => user.role === "owner")
        setOwners(ownerUsers)
      }
    } catch (error) {
      console.error("[v0] Error fetching owners:", error)
    }
  }

  const handleCreateBranch = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!newBranch.name || !newBranch.address) {
      toast({
        title: "Error",
        description: "Please fill in all required fields",
        variant: "destructive",
      })
      return
    }

    setCreateLoading(true)
    try {
      const response = await fetch("/api/branches", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(newBranch),
      })

      const result = await response.json()
      if (result.success) {
        toast({
          title: "Success",
          description: "Branch created successfully!",
        })
        setCreateDialogOpen(false)
        setNewBranch({ name: "", address: "", country: "", location: "", ownerId: "" })
        fetchBranches() // Refresh the list
      } else {
        toast({
          title: "Error",
          description: result.error || "Failed to create branch",
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error creating branch:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    } finally {
      setCreateLoading(false)
    }
  }

  const handleDeleteBranch = async (branchId: string) => {
    if (!confirm("Are you sure you want to delete this branch?")) return

    try {
      const response = await fetch(`/api/branches/${branchId}`, {
        method: "DELETE",
      })
      const result = await response.json()

      if (result.success) {
        setBranches(branches.filter((branch) => branch._id !== branchId))
        console.log("[v0] Branch deleted successfully")
        toast({
          title: "Success",
          description: "Branch deleted successfully!",
        })
      } else {
        toast({
          title: "Error",
          description: result.error || "Failed to delete branch",
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error deleting branch:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  return (
    <DashboardLayout>
      <div className="p-3 sm:p-4 lg:p-6">
        <div className="mx-auto max-w-7xl space-y-4 sm:space-y-6">
          {/* Breadcrumb */}
          <Breadcrumb>
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink href="/">Dashboard</BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbPage>All Branch List</BreadcrumbPage>
              </BreadcrumbItem>
            </BreadcrumbList>
          </Breadcrumb>

          {/* Branch List Card */}
          <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-slate-50/50 hover:shadow-xl transition-all duration-300">
            <CardHeader className="pb-4 sm:pb-6">
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center gap-3 text-xl sm:text-2xl">
                  <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center shadow-lg">
                    <Building className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                  </div>
                  Branch List ({branches.length} branches)
                </CardTitle>
                <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
                  <DialogTrigger asChild>
                    <Button className="bg-blue-600 hover:bg-blue-700 text-white">
                      <Plus className="h-4 w-4 mr-2" />
                      Create Branch
                    </Button>
                  </DialogTrigger>
                  <DialogContent className="sm:max-w-[425px]">
                    <DialogHeader>
                      <DialogTitle>Create New Branch</DialogTitle>
                      <DialogDescription>
                        Add a new branch to your organization. Fill in the required information below.
                      </DialogDescription>
                    </DialogHeader>
                    <form onSubmit={handleCreateBranch}>
                      <div className="grid gap-4 py-4">
                        <div className="space-y-2">
                          <Label htmlFor="name">Branch Name *</Label>
                          <Input
                            id="name"
                            value={newBranch.name}
                            onChange={(e) => setNewBranch({ ...newBranch, name: e.target.value })}
                            placeholder="Enter branch name"
                            required
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="address">Address *</Label>
                          <Input
                            id="address"
                            value={newBranch.address}
                            onChange={(e) => setNewBranch({ ...newBranch, address: e.target.value })}
                            placeholder="Enter branch address"
                            required
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="country">Country</Label>
                          <Input
                            id="country"
                            value={newBranch.country}
                            onChange={(e) => setNewBranch({ ...newBranch, country: e.target.value })}
                            placeholder="Enter country"
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="location">Location</Label>
                          <Input
                            id="location"
                            value={newBranch.location}
                            onChange={(e) => setNewBranch({ ...newBranch, location: e.target.value })}
                            placeholder="Enter location"
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="owner">Owner</Label>
                          <Select
                            value={newBranch.ownerId}
                            onValueChange={(value) => setNewBranch({ ...newBranch, ownerId: value })}
                          >
                            <SelectTrigger>
                              <SelectValue placeholder="Select owner (optional)" />
                            </SelectTrigger>
                            <SelectContent>
                              {owners.map((owner) => (
                                <SelectItem key={owner._id} value={owner._id}>
                                  {owner.name} ({owner.email})
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                        </div>
                      </div>
                      <DialogFooter>
                        <Button type="button" variant="outline" onClick={() => setCreateDialogOpen(false)}>
                          Cancel
                        </Button>
                        <Button type="submit" disabled={createLoading}>
                          {createLoading ? (
                            <>
                              <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                              Creating...
                            </>
                          ) : (
                            "Create Branch"
                          )}
                        </Button>
                      </DialogFooter>
                    </form>
                  </DialogContent>
                </Dialog>
              </div>
            </CardHeader>
            <CardContent className="px-3 sm:px-4 md:px-6">
              {loading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="h-6 w-6 animate-spin mr-2" />
                  Loading branches from database...
                </div>
              ) : error ? (
                <div className="text-center py-8 text-red-600">Error loading branches: {error}</div>
              ) : (
                <div className="overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="w-20">Serial No</TableHead>
                        <TableHead>Branch</TableHead>
                        <TableHead>Address</TableHead>
                        <TableHead>Country</TableHead>
                        <TableHead>Location</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead>Created At</TableHead>
                        <TableHead className="text-center min-w-[300px]">Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {branches.length === 0 ? (
                        <TableRow>
                          <TableCell colSpan={8} className="text-center py-8 text-gray-500">
                            No branches found in database
                          </TableCell>
                        </TableRow>
                      ) : (
                        branches.map((branch, index) => (
                          <TableRow key={branch._id}>
                            <TableCell className="font-medium">{index + 1}</TableCell>
                            <TableCell className="font-medium">{branch.name}</TableCell>
                            <TableCell>{branch.address}</TableCell>
                            <TableCell>{branch.country || "-"}</TableCell>
                            <TableCell>{branch.location || "-"}</TableCell>
                            <TableCell>
                              <span
                                className={`px-2 py-1 rounded-full text-xs ${
                                  branch.isActive ? "bg-green-100 text-green-800" : "bg-red-100 text-red-800"
                                }`}
                              >
                                {branch.isActive ? "Active" : "Inactive"}
                              </span>
                            </TableCell>
                            <TableCell>{new Date(branch.createdAt).toLocaleDateString()}</TableCell>
                            <TableCell>
                              <div className="flex flex-wrap gap-1 sm:gap-2 justify-center">
                                <Link href={`/edit-branch/${branch._id}`}>
                                  <Button size="sm" variant="outline" className="h-7 px-2 text-xs bg-transparent">
                                    <Edit className="h-3 w-3 mr-1" />
                                    Edit
                                  </Button>
                                </Link>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 px-2 text-xs text-red-600 border-red-200 hover:bg-red-50 bg-transparent"
                                  onClick={() => handleDeleteBranch(branch._id)}
                                >
                                  <Trash2 className="h-3 w-3 mr-1" />
                                  Delete
                                </Button>
                                <Link href={`/reports?branchId=${branch._id}`}>
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="h-7 px-2 text-xs text-green-600 border-green-200 hover:bg-green-50 bg-transparent"
                                  >
                                    <FileBarChart className="h-3 w-3 mr-1" />
                                    Report
                                  </Button>
                                </Link>
                                <Link href={`/sales-report?branchId=${branch._id}`}>
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="h-7 px-2 text-xs text-blue-600 border-blue-200 hover:bg-blue-50 bg-transparent"
                                  >
                                    <BarChart3 className="h-3 w-3 mr-1" />
                                    Total Report
                                  </Button>
                                </Link>
                                <Link href={`/expense-report?branchId=${branch._id}`}>
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="h-7 px-2 text-xs text-purple-600 border-purple-200 hover:bg-purple-50 bg-transparent"
                                  >
                                    <Search className="h-3 w-3 mr-1" />
                                    Search Report
                                  </Button>
                                </Link>
                              </div>
                            </TableCell>
                          </TableRow>
                        ))
                      )}
                    </TableBody>
                  </Table>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  )
}
