"use client"

import type React from "react"
import { useState, useEffect } from "react"
import Link from "next/link"
import { Building, Users, Save, Eye, Trash2, Loader2 } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Badge } from "@/components/ui/badge"
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import DashboardLayout from "@/components/dashboard-layout"
import { useToast } from "@/hooks/use-toast"

interface Owner {
  _id: string
  name: string
  email: string
  branchIds?: string[]
  createdAt: string
}

interface Branch {
  _id: string
  name: string
  address: string
  isActive: boolean
}

export default function AssignBranchPage() {
  const [selectedOwner, setSelectedOwner] = useState("")
  const [selectedBranch, setSelectedBranch] = useState("")
  const [owners, setOwners] = useState<Owner[]>([])
  const [branches, setBranches] = useState<Branch[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [viewBranchesDialog, setViewBranchesDialog] = useState(false)
  const [selectedOwnerBranches, setSelectedOwnerBranches] = useState<Branch[]>([])
  const [selectedOwnerForAction, setSelectedOwnerForAction] = useState<Owner | null>(null)
  const { toast } = useToast()

  const user = {
    name: "John Doe",
    email: "john@example.com",
    avatar: "/placeholder.svg?height=40&width=40",
  }

  useEffect(() => {
    fetchData()
  }, [])

  const fetchData = async () => {
    try {
      console.log("[v0] Fetching owners and branches from database...")

      const ownersResponse = await fetch("/api/users")
      const ownersResult = await ownersResponse.json()

      // Fetch branches
      const branchesResponse = await fetch("/api/branches")
      const branchesResult = await branchesResponse.json()

      if (ownersResult.success && branchesResult.success) {
        const ownerUsers = ownersResult.data.filter((user: any) => user.role === "owner")
        setOwners(ownerUsers)
        setBranches(branchesResult.data)
        console.log("[v0] Successfully loaded data")
      } else {
        setError("Failed to fetch data")
      }
    } catch (error) {
      console.error("[v0] Error fetching data:", error)
      setError("Network error occurred")
    } finally {
      setLoading(false)
    }
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!selectedOwner || !selectedBranch) return

    try {
      console.log("[v0] Assigning branch to owner...")
      console.log("[v0] Selected owner:", selectedOwner)
      console.log("[v0] Selected branch:", selectedBranch)

      const response = await fetch(`/api/users/${selectedOwner}`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          branchId: selectedBranch,
        }),
      })

      console.log("[v0] Response status:", response.status)
      const result = await response.json()
      console.log("[v0] Response result:", result)

      if (result.success) {
        // Refresh the data
        await fetchData()
        setSelectedOwner("")
        setSelectedBranch("")
        console.log("[v0] Branch assigned successfully")
        toast({
          title: "Success",
          description: "Branch assigned successfully!",
        })
      } else {
        console.error("[v0] Assignment failed:", result.error)
        toast({
          title: "Error",
          description: result.error || "Failed to assign branch",
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error assigning branch:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  const handleViewBranches = (owner: Owner) => {
    const ownerBranches = branches.filter((branch) => owner.branchIds?.includes(branch._id))
    setSelectedOwnerBranches(ownerBranches)
    setSelectedOwnerForAction(owner)
    setViewBranchesDialog(true)
  }

  const handleRemoveBranch = async (owner: Owner, branchId: string) => {
    if (!confirm(`Are you sure you want to remove this branch assignment from ${owner.name}?`)) return

    try {
      console.log("[v0] Removing branch assignment...")
      console.log("[v0] Owner:", owner._id)
      console.log("[v0] Branch to remove:", branchId)

      const updatedBranchIds = owner.branchIds?.filter((id) => id !== branchId) || []
      console.log("[v0] Updated branch IDs:", updatedBranchIds)

      const response = await fetch(`/api/users/${owner._id}`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          branchIds: updatedBranchIds,
        }),
      })

      const result = await response.json()
      console.log("[v0] Remove branch result:", result)

      if (result.success) {
        await fetchData() // Refresh data
        const updatedOwner = owners.find((o) => o._id === owner._id)
        if (updatedOwner) {
          const ownerBranches = branches.filter((branch) => updatedOwner.branchIds?.includes(branch._id))
          setSelectedOwnerBranches(ownerBranches)
        }

        toast({
          title: "Success",
          description: "Branch assignment removed successfully!",
        })
      } else {
        toast({
          title: "Error",
          description: result.error || "Failed to remove branch assignment",
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error removing branch assignment:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  const getBranchCount = (ownerId: string) => {
    const owner = owners.find((o) => o._id === ownerId)
    return owner?.branchIds?.length || 0
  }

  if (loading) {
    return (
      <DashboardLayout user={user}>
        <div className="p-6">
          <div className="flex items-center justify-center h-64">
            <Loader2 className="h-6 w-6 animate-spin mr-2" />
            <div className="text-gray-600">Loading data from database...</div>
          </div>
        </div>
      </DashboardLayout>
    )
  }

  if (error) {
    return (
      <DashboardLayout user={user}>
        <div className="p-6">
          <div className="text-center py-8 text-red-600">Error loading data: {error}</div>
        </div>
      </DashboardLayout>
    )
  }

  return (
    <DashboardLayout user={user}>
      <div className="p-3 sm:p-4 md:p-6 lg:p-8">
        <div className="mx-auto max-w-7xl space-y-6 sm:space-y-8">
          {/* Breadcrumb */}
          <nav className="flex text-sm text-gray-600">
            <Link href="/" className="hover:text-blue-600">
              Dashboard
            </Link>
            <span className="mx-2">/</span>
            <span className="text-gray-900">Assign Branch</span>
          </nav>

          {/* Page Header */}
          <div className="flex items-center gap-3 mb-6">
            <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center shadow-lg">
              <Building className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
            </div>
            <h1 className="text-2xl sm:text-3xl font-bold text-gray-900">Assign Branch</h1>
          </div>

          {/* Assign Branch Form */}
          <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-blue-50/30 hover:shadow-xl transition-all duration-300">
            <CardHeader className="pb-4 sm:pb-6">
              <CardTitle className="flex items-center gap-3 text-xl sm:text-2xl">
                <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center">
                  <Building className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                </div>
                Assign Branch to Owner
              </CardTitle>
            </CardHeader>
            <CardContent className="px-4 sm:px-6">
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="grid gap-6 md:grid-cols-2">
                  <div className="space-y-2">
                    <Label htmlFor="owner" className="text-sm font-medium">
                      Select Owner <span className="text-red-500">*</span>
                    </Label>
                    <Select value={selectedOwner} onValueChange={setSelectedOwner}>
                      <SelectTrigger className="h-10 sm:h-11">
                        <SelectValue placeholder="Select Owner" />
                      </SelectTrigger>
                      <SelectContent>
                        {owners.map((owner) => (
                          <SelectItem key={owner._id} value={owner._id}>
                            {owner.name} ({owner.email})
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="space-y-2">
                    <Label className="text-sm font-medium">
                      Select Branch <span className="text-red-500">*</span>
                    </Label>
                    <Select value={selectedBranch} onValueChange={setSelectedBranch}>
                      <SelectTrigger className="h-10 sm:h-11">
                        <SelectValue placeholder="Select Branch" />
                      </SelectTrigger>
                      <SelectContent>
                        {branches.map((branch) => (
                          <SelectItem key={branch._id} value={branch._id}>
                            {branch.name} - {branch.address}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                <div className="flex justify-end">
                  <Button
                    type="submit"
                    className="bg-red-600 hover:bg-red-700 text-white px-6 py-2 sm:px-8 sm:py-3"
                    disabled={!selectedOwner || !selectedBranch}
                  >
                    <Save className="mr-2 h-4 w-4" />
                    Assign Branch
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>

          {/* Owner List */}
          <Card className="border-0 shadow-lg bg-white hover:shadow-xl transition-all duration-300">
            <CardHeader className="pb-4 sm:pb-6">
              <CardTitle className="flex items-center gap-3 text-xl sm:text-2xl">
                <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-green-500 to-emerald-600 flex items-center justify-center">
                  <Users className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                </div>
                Owner List ({owners.length} owners)
              </CardTitle>
            </CardHeader>
            <CardContent className="px-4 sm:px-6">
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-20">Serial No</TableHead>
                      <TableHead>Name</TableHead>
                      <TableHead>Email</TableHead>
                      <TableHead>Branches</TableHead>
                      <TableHead>Created At</TableHead>
                      <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {owners.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={6} className="text-center py-8 text-gray-500">
                          No owners found in database
                        </TableCell>
                      </TableRow>
                    ) : (
                      owners.map((owner, index) => (
                        <TableRow key={owner._id}>
                          <TableCell className="font-medium">{index + 1}</TableCell>
                          <TableCell>{owner.name}</TableCell>
                          <TableCell>{owner.email}</TableCell>
                          <TableCell>
                            <Badge variant="secondary" className="bg-blue-100 text-blue-800">
                              {getBranchCount(owner._id)} branch(es)
                            </Badge>
                          </TableCell>
                          <TableCell>{new Date(owner.createdAt).toLocaleDateString()}</TableCell>
                          <TableCell className="text-right">
                            <div className="flex justify-end gap-2">
                              <Button
                                size="sm"
                                variant="outline"
                                className="text-blue-600 border-blue-200 hover:bg-blue-50 bg-transparent"
                                onClick={() => handleViewBranches(owner)}
                                disabled={getBranchCount(owner._id) === 0}
                              >
                                <Eye className="h-3 w-3 mr-1" />
                                View Branches
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </div>
            </CardContent>
          </Card>

          <Dialog open={viewBranchesDialog} onOpenChange={setViewBranchesDialog}>
            <DialogContent className="sm:max-w-[600px]">
              <DialogHeader>
                <DialogTitle>Branches assigned to {selectedOwnerForAction?.name}</DialogTitle>
                <DialogDescription>View and manage branch assignments for this owner.</DialogDescription>
              </DialogHeader>
              <div className="py-4">
                {selectedOwnerBranches.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">No branches assigned to this owner</div>
                ) : (
                  <div className="space-y-3">
                    {selectedOwnerBranches.map((branch) => (
                      <div key={branch._id} className="flex items-center justify-between p-3 border rounded-lg">
                        <div>
                          <h4 className="font-medium">{branch.name}</h4>
                          <p className="text-sm text-gray-600">{branch.address}</p>
                        </div>
                        <Button
                          size="sm"
                          variant="outline"
                          className="text-red-600 border-red-200 hover:bg-red-50 bg-transparent"
                          onClick={() => {
                            if (selectedOwnerForAction) {
                              handleRemoveBranch(selectedOwnerForAction, branch._id)
                            }
                          }}
                        >
                          <Trash2 className="h-3 w-3 mr-1" />
                          Remove
                        </Button>
                      </div>
                    ))}
                  </div>
                )}
              </div>
            </DialogContent>
          </Dialog>
        </div>
      </div>
    </DashboardLayout>
  )
}
