import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import bcrypt from "bcryptjs"
import { getUsersCollection } from "@/lib/database/collections"
import type { CreateUserInput, User } from "@/lib/models/User"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const role = searchParams.get("role")
    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "10")
    const active = searchParams.get("active")

    const usersCollection = await getUsersCollection()

    // Build query
    const query: any = {}
    if (role) {
      query.role = role
    }
    if (active !== null) {
      query.isActive = active === "true"
    }

    // Get total count for pagination
    const total = await usersCollection.countDocuments(query)

    // Get users data with pagination (exclude password)
    const users = await usersCollection
      .find(query, { projection: { password: 0 } })
      .sort({ createdAt: -1 })
      .skip((page - 1) * limit)
      .limit(limit)
      .toArray()

    return NextResponse.json({
      success: true,
      data: users,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    })
  } catch (error) {
    console.error("Error fetching users:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch users data" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body: CreateUserInput = await request.json()

    // Check if user already exists
    const usersCollection = await getUsersCollection()
    const existingUser = await usersCollection.findOne({ email: body.email })

    if (existingUser) {
      return NextResponse.json({ success: false, error: "User with this email already exists" }, { status: 400 })
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(body.password, 12)

    const userData: Omit<User, "_id"> = {
      name: body.name,
      email: body.email,
      phone: body.phone,
      password: hashedPassword,
      role: body.role,
      branchIds: body.branchIds?.map((id) => new ObjectId(id)),
      isActive: true,
      createdAt: new Date(),
      updatedAt: new Date(),
    }

    const result = await usersCollection.insertOne(userData)

    // Return user data without password
    const { password, ...userResponse } = userData
    return NextResponse.json({
      success: true,
      data: { _id: result.insertedId, ...userResponse },
    })
  } catch (error) {
    console.error("Error creating user:", error)
    return NextResponse.json({ success: false, error: "Failed to create user" }, { status: 500 })
  }
}
