import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getSalesCollection, getBranchesCollection } from "@/lib/database/collections"
import type { CreateSalesInput, Sales } from "@/lib/models/Sales"
import { calculateSalesTotals } from "@/lib/utils/calculations"
import { WebSocketEvents } from "@/lib/websocket/events"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const branchId = searchParams.get("branchId")
    const fromDate = searchParams.get("fromDate")
    const toDate = searchParams.get("toDate")
    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "10")

    const salesCollection = await getSalesCollection()

    // Build query
    const query: any = {}
    if (branchId && branchId !== "all") {
      query.branchId = new ObjectId(branchId)
    }
    if (fromDate || toDate) {
      query.date = {}
      if (fromDate) query.date.$gte = new Date(fromDate)
      if (toDate) query.date.$lte = new Date(toDate)
    }

    // Get total count for pagination
    const total = await salesCollection.countDocuments(query)

    // Get sales data with pagination
    const sales = await salesCollection
      .find(query)
      .sort({ date: -1 })
      .skip((page - 1) * limit)
      .limit(limit)
      .toArray()

    // Populate branch names
    const branchesCollection = await getBranchesCollection()
    const salesWithBranches = await Promise.all(
      sales.map(async (sale) => {
        const branch = await branchesCollection.findOne({ _id: sale.branchId })
        return {
          ...sale,
          branchName: branch?.name || "Unknown Branch",
        }
      }),
    )

    return NextResponse.json({
      success: true,
      data: salesWithBranches,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    })
  } catch (error) {
    console.error("Error fetching sales:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch sales data" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body: CreateSalesInput = await request.json()

    // Calculate totals
    const totals = calculateSalesTotals(body)

    const salesData: Omit<Sales, "_id"> = {
      date: new Date(body.date),
      branchId: new ObjectId(body.branchId),
      cashInShop: body.cashInShop,
      cashDelivery: body.cashDelivery,
      cashWolt: body.cashWolt,
      cashBolt: body.cashBolt,
      cashPyszne: body.cashPyszne,
      cashUberEats: body.cashUberEats,
      cashGlovo: body.cashGlovo,
      cardInShop: body.cardInShop,
      cardDelivery: body.cardDelivery,
      cardWolt: body.cardWolt,
      cardBolt: body.cardBolt,
      cardPyszne: body.cardPyszne,
      cardUberEats: body.cardUberEats,
      cardGlovo: body.cardGlovo,
      totalCash: totals.totalCash,
      totalCard: totals.totalCard,
      totalSales: totals.totalSales,
      notes: body.notes,
      createdBy: new ObjectId(), // TODO: Get from auth session
      createdAt: new Date(),
      updatedAt: new Date(),
    }

    const salesCollection = await getSalesCollection()
    const result = await salesCollection.insertOne(salesData)

    await WebSocketEvents.onSalesCreate({
      _id: result.insertedId,
      ...salesData,
      totalAmount: totals.totalSales,
    })

    return NextResponse.json({
      success: true,
      data: { _id: result.insertedId, ...salesData },
    })
  } catch (error) {
    console.error("Error creating sales:", error)

    await WebSocketEvents.onError("Failed to create sales record")

    return NextResponse.json({ success: false, error: "Failed to create sales record" }, { status: 500 })
  }
}
