import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getInventoryCollection, getBranchesCollection } from "@/lib/database/collections"
import type { CreateInventoryInput, Inventory } from "@/lib/models/Inventory"
import { WebSocketEvents } from "@/lib/websocket/events"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const branchId = searchParams.get("branchId")
    const type = searchParams.get("type")
    const fromDate = searchParams.get("fromDate")
    const toDate = searchParams.get("toDate")
    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "10")

    const inventoryCollection = await getInventoryCollection()

    // Build query
    const query: any = {}
    if (branchId && branchId !== "all") {
      query.branchId = new ObjectId(branchId)
    }
    if (type) {
      query.type = type
    }
    if (fromDate || toDate) {
      query.date = {}
      if (fromDate) query.date.$gte = new Date(fromDate)
      if (toDate) query.date.$lte = new Date(toDate)
    }

    // Get total count for pagination
    const total = await inventoryCollection.countDocuments(query)

    // Get inventory data with pagination
    const inventory = await inventoryCollection
      .find(query)
      .sort({ date: -1 })
      .skip((page - 1) * limit)
      .limit(limit)
      .toArray()

    // Populate branch names
    const branchesCollection = await getBranchesCollection()
    const inventoryWithBranches = await Promise.all(
      inventory.map(async (item) => {
        const branch = await branchesCollection.findOne({ _id: item.branchId })
        return {
          ...item,
          branchName: branch?.name || "Unknown Branch",
        }
      }),
    )

    return NextResponse.json({
      success: true,
      data: inventoryWithBranches,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    })
  } catch (error) {
    console.error("Error fetching inventory:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch inventory data" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body: CreateInventoryInput = await request.json()

    const inventoryData: Omit<Inventory, "_id"> = {
      date: new Date(body.date),
      branchId: new ObjectId(body.branchId),
      type: body.type,
      kurczak: body.kurczak,
      wolowina: body.wolowina,
      kraftowe: body.kraftowe,
      majonez: body.majonez,
      ketchup: body.ketchup,
      smabal: body.smabal,
      yogurt: body.yogurt,
      pita20cm: body.pita20cm,
      pita30cm: body.pita30cm,
      gas: body.gas,
      notes: body.notes,
      createdBy: new ObjectId(), // TODO: Get from auth session
      createdAt: new Date(),
      updatedAt: new Date(),
    }

    const inventoryCollection = await getInventoryCollection()
    const result = await inventoryCollection.insertOne(inventoryData)

    try {
      await WebSocketEvents.onInventoryUpdate({
        _id: result.insertedId,
        ...inventoryData,
        itemName: `${body.type} inventory`,
      })
    } catch (wsError) {
      console.error("WebSocket notification failed:", wsError)
      // Continue with response even if WebSocket fails
    }

    return NextResponse.json({
      success: true,
      data: { _id: result.insertedId, ...inventoryData },
    })
  } catch (error) {
    console.error("Error creating inventory:", error)

    try {
      await WebSocketEvents.onError("Failed to create inventory record")
    } catch (wsError) {
      console.error("WebSocket error notification failed:", wsError)
    }

    return NextResponse.json({ success: false, error: "Failed to create inventory record" }, { status: 500 })
  }
}
