import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getExpensesCollection, getBranchesCollection, getEmployeesCollection } from "@/lib/database/collections"
import type { CreateExpenseInput, Expense } from "@/lib/models/Expense"
import { calculateSalaryTotals, calculateTaxTotals } from "@/lib/utils/calculations"
import { WebSocketEvents } from "@/lib/websocket/events"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const branchId = searchParams.get("branchId")
    const category = searchParams.get("category")
    const fromDate = searchParams.get("fromDate")
    const toDate = searchParams.get("toDate")
    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "10")

    const expensesCollection = await getExpensesCollection()

    // Build query
    const query: any = {}
    if (branchId && branchId !== "all") {
      query.branchId = new ObjectId(branchId)
    }
    if (category) {
      query.category = category
    }
    if (fromDate || toDate) {
      query.date = {}
      if (fromDate) query.date.$gte = new Date(fromDate)
      if (toDate) query.date.$lte = new Date(toDate)
    }

    // Get total count for pagination
    const total = await expensesCollection.countDocuments(query)

    // Get expenses data with pagination
    const expenses = await expensesCollection
      .find(query)
      .sort({ date: -1 })
      .skip((page - 1) * limit)
      .limit(limit)
      .toArray()

    const branchesCollection = await getBranchesCollection()
    const employeesCollection = await getEmployeesCollection()

    const expensesWithPopulatedData = await Promise.all(
      expenses.map(async (expense) => {
        const branch = await branchesCollection.findOne({ _id: expense.branchId })
        let employeeName = null

        if (expense.employeeId) {
          const employee = await employeesCollection.findOne({ _id: expense.employeeId })
          employeeName = employee ? employee.name : "Unknown Employee"
        }

        return {
          ...expense,
          branchName: branch?.name || "Unknown Branch",
          employeeName,
        }
      }),
    )

    return NextResponse.json({
      success: true,
      data: expensesWithPopulatedData,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    })
  } catch (error) {
    console.error("Error fetching expenses:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch expenses data" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body: CreateExpenseInput = await request.json()

    console.log("[v0] Received expense data:", body)

    let branchObjectId: ObjectId

    // If branchId is already a valid ObjectId string, use it directly
    if (ObjectId.isValid(body.branchId) && body.branchId.length === 24) {
      branchObjectId = new ObjectId(body.branchId)
    } else {
      // If branchId is a branch name (like "lablu" or "gulshan"), find the actual branch ObjectId
      const branchesCollection = await getBranchesCollection()
      const branch = await branchesCollection.findOne({
        $or: [{ name: body.branchId }, { slug: body.branchId }, { _id: body.branchId }],
      })

      if (!branch) {
        console.error("[v0] Branch not found for branchId:", body.branchId)
        return NextResponse.json(
          {
            success: false,
            error: `Branch not found: ${body.branchId}`,
          },
          { status: 400 },
        )
      }

      branchObjectId = branch._id
    }

    console.log("[v0] Using branch ObjectId:", branchObjectId)

    // Calculate salary totals if applicable
    const salaryTotals = calculateSalaryTotals({
      hourlyRate: body.hourlyRate,
      totalHours: body.totalHours,
      received: body.received,
    })

    // Calculate tax totals if applicable
    const taxTotals = calculateTaxTotals({
      netto: body.netto,
      vat: body.vat,
    })

    const expenseData: Omit<Expense, "_id"> = {
      date: new Date(body.date),
      branchId: branchObjectId, // Use validated ObjectId
      category: body.category,
      subcategory: body.subcategory,
      amount: body.amount,
      description: body.description,
      paymentMethod: body.paymentMethod,
      employeeId: body.employeeId && ObjectId.isValid(body.employeeId) ? new ObjectId(body.employeeId) : undefined,
      hourlyRate: body.hourlyRate,
      totalHours: body.totalHours,
      totalSalary: salaryTotals.totalSalary || body.totalSalary,
      received: body.received,
      payable: salaryTotals.payable || body.payable,
      netto: body.netto,
      vat: body.vat,
      brutto: taxTotals.brutto || body.brutto,
      fakturaNo: body.fakturaNo,
      notes: body.notes,
      paymentDate: body.paymentDate ? new Date(body.paymentDate) : undefined,
      createdBy: new ObjectId(),
      createdAt: new Date(),
      updatedAt: new Date(),
    }

    console.log("[v0] Final expense data:", expenseData)

    const expensesCollection = await getExpensesCollection()
    const result = await expensesCollection.insertOne(expenseData)

    await WebSocketEvents.onExpenseCreate({
      _id: result.insertedId,
      ...expenseData,
      amount: body.amount,
    })

    return NextResponse.json({
      success: true,
      data: { _id: result.insertedId, ...expenseData },
    })
  } catch (error) {
    console.error("Error creating expense:", error)

    await WebSocketEvents.onError("Failed to create expense record")

    return NextResponse.json({ success: false, error: "Failed to create expense record" }, { status: 500 })
  }
}
