import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getEmployeesCollection, getBranchesCollection } from "@/lib/database/collections"
import type { CreateEmployeeInput } from "@/lib/models/Employee"

export async function GET(request: NextRequest) {
  try {
    console.log("[v0] Fetching employees from database...")
    const employeesCollection = await getEmployeesCollection()
    const branchesCollection = await getBranchesCollection()

    const searchParams = request.nextUrl.searchParams
    const branchId = searchParams.get("branchId")

    let query = {}
    if (branchId && ObjectId.isValid(branchId)) {
      query = { branchId: new ObjectId(branchId) }
    }

    const employees = await employeesCollection.find(query).sort({ createdAt: -1 }).toArray()

    const employeesWithBranchNames = await Promise.all(
      employees.map(async (employee) => {
        const branch = await branchesCollection.findOne({ _id: employee.branchId })
        return {
          ...employee,
          branchName: branch?.name || "Unknown Branch",
          status: employee.isActive ? "active" : "inactive", // Normalize status field
        }
      }),
    )

    console.log(`[v0] Successfully fetched ${employeesWithBranchNames.length} employees`)

    return NextResponse.json({
      success: true,
      data: employeesWithBranchNames,
    })
  } catch (error) {
    console.error("[v0] Error fetching employees:", error)
    return NextResponse.json(
      {
        success: false,
        error: "Failed to fetch employees",
      },
      { status: 500 },
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    console.log("[v0] Creating new employee...")
    const body: CreateEmployeeInput = await request.json()

    // Validate required fields
    if (!body.name || !body.position || !body.branchId) {
      return NextResponse.json(
        {
          success: false,
          error: "Name, position, and branch are required",
        },
        { status: 400 },
      )
    }

    // Validate branchId
    if (!ObjectId.isValid(body.branchId)) {
      return NextResponse.json(
        {
          success: false,
          error: "Invalid branch ID",
        },
        { status: 400 },
      )
    }

    const employeesCollection = await getEmployeesCollection()

    const newEmployee = {
      name: body.name,
      email: body.email || "",
      phone: body.phone || "",
      position: body.position,
      branchId: new ObjectId(body.branchId),
      hourlyRate: body.hourlyRate || 0,
      isActive: true,
      startDate: body.startDate ? new Date(body.startDate) : new Date(),
      createdAt: new Date(),
      updatedAt: new Date(),
    }

    const result = await employeesCollection.insertOne(newEmployee)

    const createdEmployee = await employeesCollection.findOne({ _id: result.insertedId })

    console.log("[v0] Employee created successfully:", result.insertedId)

    return NextResponse.json({
      success: true,
      data: createdEmployee,
    })
  } catch (error) {
    console.error("[v0] Error creating employee:", error)
    return NextResponse.json(
      {
        success: false,
        error: "Failed to create employee",
      },
      { status: 500 },
    )
  }
}
