import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getSalesCollection, getExpensesCollection, getBranchesCollection } from "@/lib/database/collections"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const branchId = searchParams.get("branchId")
    const fromDate = searchParams.get("fromDate")
    const toDate = searchParams.get("toDate")

    // Build date filter
    const dateFilter: any = {}
    if (fromDate || toDate) {
      dateFilter.date = {}
      if (fromDate) dateFilter.date.$gte = new Date(fromDate)
      if (toDate) dateFilter.date.$lte = new Date(toDate)
    }

    // Build branch filter
    const branchFilter: any = {}
    if (branchId && branchId !== "all") {
      branchFilter.branchId = new ObjectId(branchId)
    }

    const salesCollection = await getSalesCollection()
    const expensesCollection = await getExpensesCollection()
    const branchesCollection = await getBranchesCollection()

    // Get sales metrics
    const salesAggregation = await salesCollection
      .aggregate([
        { $match: { ...dateFilter, ...branchFilter } },
        {
          $group: {
            _id: null,
            totalSales: { $sum: "$totalSales" },
            totalCash: { $sum: "$totalCash" },
            totalCard: { $sum: "$totalCard" },
            count: { $sum: 1 },
          },
        },
      ])
      .toArray()

    // Get expenses metrics
    const expensesAggregation = await expensesCollection
      .aggregate([
        { $match: { ...dateFilter, ...branchFilter } },
        {
          $group: {
            _id: null,
            totalExpenses: { $sum: "$amount" },
            count: { $sum: 1 },
          },
        },
      ])
      .toArray()

    // Get active branches count
    const activeBranches = await branchesCollection.countDocuments({ isActive: true })

    // Calculate metrics
    const salesData = salesAggregation[0] || { totalSales: 0, totalCash: 0, totalCard: 0, count: 0 }
    const expensesData = expensesAggregation[0] || { totalExpenses: 0, count: 0 }
    const netProfit = salesData.totalSales - expensesData.totalExpenses

    // Get monthly data for charts
    const monthlyData = await salesCollection
      .aggregate([
        { $match: { ...branchFilter } },
        {
          $group: {
            _id: {
              year: { $year: "$date" },
              month: { $month: "$date" },
            },
            sales: { $sum: "$totalSales" },
            expenses: { $sum: 0 }, // Will be populated separately
          },
        },
        { $sort: { "_id.year": 1, "_id.month": 1 } },
        { $limit: 12 },
      ])
      .toArray()

    // Add expenses to monthly data
    const monthlyExpenses = await expensesCollection
      .aggregate([
        { $match: { ...branchFilter } },
        {
          $group: {
            _id: {
              year: { $year: "$date" },
              month: { $month: "$date" },
            },
            expenses: { $sum: "$amount" },
          },
        },
      ])
      .toArray()

    // Merge sales and expenses data
    const monthlyDataWithExpenses = monthlyData.map((item) => {
      const expenseItem = monthlyExpenses.find(
        (exp) => exp._id.year === item._id.year && exp._id.month === item._id.month,
      )
      return {
        month: new Date(item._id.year, item._id.month - 1).toLocaleDateString("en-US", { month: "short" }),
        sales: item.sales,
        expenses: expenseItem?.expenses || 0,
      }
    })

    return NextResponse.json({
      success: true,
      data: {
        metrics: {
          totalSales: { value: salesData.totalSales, change: 0, trend: "neutral" },
          totalExpenses: { value: expensesData.totalExpenses, change: 0, trend: "neutral" },
          netProfit: { value: netProfit, change: 0, trend: netProfit >= 0 ? "up" : "down" },
          activeBranches: { value: activeBranches, change: 0, trend: "neutral" },
        },
        monthlyData: monthlyDataWithExpenses,
      },
    })
  } catch (error) {
    console.error("Error fetching dashboard metrics:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch dashboard metrics" }, { status: 500 })
  }
}
