import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getBranchesCollection, getUsersCollection } from "@/lib/database/collections"
import type { CreateBranchInput, Branch } from "@/lib/models/Branch"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "10")
    const active = searchParams.get("active")

    const branchesCollection = await getBranchesCollection()

    // Build query
    const query: any = {}
    if (active !== null) {
      query.isActive = active === "true"
    }

    // Get total count for pagination
    const total = await branchesCollection.countDocuments(query)

    // Get branches data with pagination
    const branches = await branchesCollection
      .find(query)
      .sort({ createdAt: -1 })
      .skip((page - 1) * limit)
      .limit(limit)
      .toArray()

    // Populate owner and manager names
    const usersCollection = await getUsersCollection()
    const branchesWithUsers = await Promise.all(
      branches.map(async (branch) => {
        const owner = branch.ownerId ? await usersCollection.findOne({ _id: branch.ownerId }) : null
        const manager = branch.managerId ? await usersCollection.findOne({ _id: branch.managerId }) : null

        return {
          ...branch,
          ownerName: owner?.name || null,
          managerName: manager?.name || null,
        }
      }),
    )

    return NextResponse.json({
      success: true,
      data: branchesWithUsers,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    })
  } catch (error) {
    console.error("Error fetching branches:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch branches data" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body: CreateBranchInput = await request.json()

    const branchData: Omit<Branch, "_id"> = {
      name: body.name,
      address: body.address,
      country: body.country,
      location: body.location,
      ownerId: body.ownerId ? new ObjectId(body.ownerId) : undefined,
      managerId: body.managerId ? new ObjectId(body.managerId) : undefined,
      isActive: true,
      createdAt: new Date(),
      updatedAt: new Date(),
    }

    const branchesCollection = await getBranchesCollection()
    const result = await branchesCollection.insertOne(branchData)

    return NextResponse.json({
      success: true,
      data: { _id: result.insertedId, ...branchData },
    })
  } catch (error) {
    console.error("Error creating branch:", error)
    return NextResponse.json({ success: false, error: "Failed to create branch" }, { status: 500 })
  }
}
