import { type NextRequest, NextResponse } from "next/server"
import bcrypt from "bcryptjs"
import { ObjectId } from "mongodb"
import { getUsersCollection } from "@/lib/database/collections"
import { createSession } from "@/lib/auth/session-simple"
import type { CreateUserInput, User } from "@/lib/models/User"

export async function POST(request: NextRequest) {
  try {
    const body: CreateUserInput = await request.json()

    if (!body.name || !body.email || !body.password) {
      return NextResponse.json({ success: false, error: "Name, email, and password are required" }, { status: 400 })
    }

    const usersCollection = await getUsersCollection()

    // Check if user already exists
    const existingUser = await usersCollection.findOne({ email: body.email })
    if (existingUser) {
      return NextResponse.json({ success: false, error: "User with this email already exists" }, { status: 400 })
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(body.password, 12)

    const userData: Omit<User, "_id"> = {
      name: body.name,
      email: body.email,
      phone: body.phone || "",
      password: hashedPassword,
      role: body.role || "manager", // Default role
      branchIds: body.branchIds?.map((id) => new ObjectId(id)),
      isActive: true,
      createdAt: new Date(),
      updatedAt: new Date(),
    }

    const result = await usersCollection.insertOne(userData)

    // Create session for new user
    await createSession(result.insertedId.toString(), userData.email, userData.role)

    // Return user data without password
    const { password, ...userResponse } = userData
    return NextResponse.json({
      success: true,
      data: { _id: result.insertedId, ...userResponse },
      message: "Registration successful",
    })
  } catch (error) {
    console.error("Registration error:", error)
    return NextResponse.json({ success: false, error: "Internal server error" }, { status: 500 })
  }
}
