"use client"

import type React from "react"
import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import {
  Breadcrumb,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbList,
  BreadcrumbPage,
  BreadcrumbSeparator,
} from "@/components/ui/breadcrumb"
import { UserPlus, Users, Edit, Trash2, Loader2, Eye } from "lucide-react"
import DashboardLayout from "@/components/dashboard-layout"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { useToast } from "@/hooks/use-toast"

interface Admin {
  _id: string
  name: string
  email: string
  phone: string
  role: string
  createdAt: string
}

export default function AdminPage() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    password: "",
  })

  const [admins, setAdmins] = useState<Admin[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [isViewModalOpen, setIsViewModalOpen] = useState(false)
  const [isEditModalOpen, setIsEditModalOpen] = useState(false)
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false)
  const [selectedAdmin, setSelectedAdmin] = useState<Admin | null>(null)
  const { toast } = useToast()

  useEffect(() => {
    fetchAdmins()
  }, [])

  const fetchAdmins = async () => {
    try {
      console.log("[v0] Fetching admins from database...")
      const response = await fetch("/api/users?role=admin")
      const result = await response.json()
      if (result.success) {
        setAdmins(result.data)
        console.log("[v0] Successfully loaded admins:", result.data.length)
      } else {
        setError(result.error || "Failed to fetch admins")
      }
    } catch (error) {
      console.error("[v0] Error fetching admins:", error)
      setError("Network error occurred")
    } finally {
      setLoading(false)
    }
  }

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value,
    })
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    try {
      console.log("[v0] Creating new admin...")
      const response = await fetch("/api/users", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...formData,
          role: "admin",
        }),
      })

      const result = await response.json()
      if (result.success) {
        setAdmins([result.data, ...admins])
        setFormData({ name: "", email: "", phone: "", password: "" })
        toast({
          title: "Success",
          description: "Admin created successfully!",
          variant: "default",
        })
        console.log("[v0] Admin created successfully")
      } else {
        toast({
          title: "Error",
          description: "Failed to create admin: " + result.error,
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error creating admin:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  const handleViewAdmin = (admin: Admin) => {
    setSelectedAdmin(admin)
    setIsViewModalOpen(true)
  }

  const handleEditAdmin = (admin: Admin) => {
    setSelectedAdmin(admin)
    setIsEditModalOpen(true)
  }

  const handleDeleteAdmin = async (adminId: string) => {
    try {
      const response = await fetch(`/api/users/${adminId}`, {
        method: "DELETE",
      })
      const result = await response.json()

      if (result.success) {
        setAdmins(admins.filter((admin) => admin._id !== adminId))
        toast({
          title: "Success",
          description: "Admin deleted successfully!",
          variant: "default",
        })
        setIsDeleteModalOpen(false)
        console.log("[v0] Admin deleted successfully")
      } else {
        toast({
          title: "Error",
          description: "Failed to delete admin: " + result.error,
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error deleting admin:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  const handleSaveEdit = async () => {
    if (!selectedAdmin) return

    try {
      const response = await fetch(`/api/users/${selectedAdmin._id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(selectedAdmin),
      })

      const result = await response.json()
      if (result.success) {
        setAdmins(admins.map((admin) => (admin._id === selectedAdmin._id ? result.data : admin)))
        setIsEditModalOpen(false)
        setSelectedAdmin(null)
        toast({
          title: "Success",
          description: "Admin updated successfully!",
          variant: "default",
        })
      } else {
        toast({
          title: "Error",
          description: "Failed to update admin: " + result.error,
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("[v0] Error updating admin:", error)
      toast({
        title: "Error",
        description: "Network error occurred",
        variant: "destructive",
      })
    }
  }

  return (
    <DashboardLayout>
      <div className="p-3 sm:p-4 lg:p-6">
        <div className="mx-auto max-w-7xl space-y-4 sm:space-y-6">
          {/* Breadcrumb */}
          <Breadcrumb>
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink href="/">Dashboard</BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbPage>Add Admin</BreadcrumbPage>
              </BreadcrumbItem>
            </BreadcrumbList>
          </Breadcrumb>

          {/* Add Admin Form */}
          <Card className="border-0 shadow-lg bg-gradient-to-br from-red-50 to-pink-50/50 hover:shadow-xl transition-all duration-300">
            <CardHeader className="pb-4 sm:pb-6">
              <CardTitle className="flex items-center gap-3 text-xl sm:text-2xl">
                <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-red-500 to-pink-600 flex items-center justify-center shadow-lg">
                  <UserPlus className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                </div>
                Add Admin
              </CardTitle>
            </CardHeader>
            <CardContent className="px-3 sm:px-4 md:px-6">
              <form onSubmit={handleSubmit} className="space-y-4 sm:space-y-6">
                <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 sm:gap-6">
                  <div className="lg:col-span-4 flex items-center">
                    <Label htmlFor="name" className="text-sm font-medium">
                      Name <span className="text-red-500">*</span>
                    </Label>
                  </div>
                  <div className="lg:col-span-6">
                    <Input
                      id="name"
                      name="name"
                      type="text"
                      placeholder="Enter Name."
                      value={formData.name}
                      onChange={handleInputChange}
                      required
                      className="w-full"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 sm:gap-6">
                  <div className="lg:col-span-4 flex items-center">
                    <Label htmlFor="email" className="text-sm font-medium">
                      Email <span className="text-red-500">*</span>
                    </Label>
                  </div>
                  <div className="lg:col-span-6">
                    <Input
                      id="email"
                      name="email"
                      type="email"
                      placeholder="Enter email."
                      value={formData.email}
                      onChange={handleInputChange}
                      required
                      className="w-full"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 sm:gap-6">
                  <div className="lg:col-span-4 flex items-center">
                    <Label htmlFor="phone" className="text-sm font-medium">
                      Phone <span className="text-red-500">*</span>
                    </Label>
                  </div>
                  <div className="lg:col-span-6">
                    <Input
                      id="phone"
                      name="phone"
                      type="tel"
                      placeholder="Enter phone."
                      value={formData.phone}
                      onChange={handleInputChange}
                      required
                      className="w-full"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 sm:gap-6">
                  <div className="lg:col-span-4 flex items-center">
                    <Label htmlFor="password" className="text-sm font-medium">
                      Password <span className="text-red-500">*</span>
                    </Label>
                  </div>
                  <div className="lg:col-span-6">
                    <Input
                      id="password"
                      name="password"
                      type="password"
                      placeholder="Enter Password"
                      value={formData.password}
                      onChange={handleInputChange}
                      required
                      className="w-full"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 sm:gap-6">
                  <div className="lg:col-span-8 lg:col-start-5">
                    <Button type="submit" className="bg-red-600 hover:bg-red-700 text-white px-6 py-2">
                      Create Admin
                    </Button>
                  </div>
                </div>
              </form>
            </CardContent>
          </Card>

          {/* Admin List */}
          <Card className="border-0 shadow-lg bg-gradient-to-br from-white to-slate-50/50 hover:shadow-xl transition-all duration-300">
            <CardHeader className="pb-4 sm:pb-6">
              <CardTitle className="flex items-center gap-3 text-xl sm:text-2xl">
                <div className="h-10 w-10 sm:h-12 sm:w-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center shadow-lg">
                  <Users className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                </div>
                Admin List ({admins.length} admins)
              </CardTitle>
            </CardHeader>
            <CardContent className="px-3 sm:px-4 md:px-6">
              {loading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="h-6 w-6 animate-spin mr-2" />
                  Loading admins from database...
                </div>
              ) : error ? (
                <div className="text-center py-8 text-red-600">Error loading admins: {error}</div>
              ) : (
                <div className="overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="w-20">Serial No</TableHead>
                        <TableHead>Name</TableHead>
                        <TableHead>Email</TableHead>
                        <TableHead>Phone</TableHead>
                        <TableHead>Created At</TableHead>
                        <TableHead className="text-center min-w-[150px]">Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {admins.length === 0 ? (
                        <TableRow>
                          <TableCell colSpan={6} className="text-center py-8 text-gray-500">
                            No admins found in database
                          </TableCell>
                        </TableRow>
                      ) : (
                        admins.map((admin, index) => (
                          <TableRow key={admin._id}>
                            <TableCell className="font-medium">{index + 1}</TableCell>
                            <TableCell className="font-medium">{admin.name}</TableCell>
                            <TableCell>{admin.email}</TableCell>
                            <TableCell>{admin.phone}</TableCell>
                            <TableCell>{new Date(admin.createdAt).toLocaleDateString()}</TableCell>
                            <TableCell>
                              <div className="flex flex-wrap gap-1 sm:gap-2 justify-center">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 px-2 text-xs text-green-600 border-green-200 hover:bg-green-50 bg-transparent"
                                  onClick={() => handleViewAdmin(admin)}
                                >
                                  <Eye className="h-3 w-3 mr-1" />
                                  View
                                </Button>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 px-2 text-xs text-blue-600 border-blue-200 hover:bg-blue-50 bg-transparent"
                                  onClick={() => handleEditAdmin(admin)}
                                >
                                  <Edit className="h-3 w-3 mr-1" />
                                  Edit
                                </Button>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="h-7 px-2 text-xs text-red-600 border-red-200 hover:bg-red-50 bg-transparent"
                                  onClick={() => {
                                    setSelectedAdmin(admin)
                                    setIsDeleteModalOpen(true)
                                  }}
                                >
                                  <Trash2 className="h-3 w-3 mr-1" />
                                  Delete
                                </Button>
                              </div>
                            </TableCell>
                          </TableRow>
                        ))
                      )}
                    </TableBody>
                  </Table>
                </div>
              )}
            </CardContent>
          </Card>

          {/* View Admin Modal */}
          <Dialog open={isViewModalOpen} onOpenChange={setIsViewModalOpen}>
            <DialogContent className="max-w-md">
              <DialogHeader>
                <DialogTitle>Admin Details</DialogTitle>
              </DialogHeader>
              {selectedAdmin && (
                <div className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Name</Label>
                      <p className="text-sm font-medium">{selectedAdmin.name}</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Role</Label>
                      <p className="text-sm capitalize">{selectedAdmin.role}</p>
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Email</Label>
                      <p className="text-sm">{selectedAdmin.email}</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Phone</Label>
                      <p className="text-sm">{selectedAdmin.phone}</p>
                    </div>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Created</Label>
                    <p className="text-sm">{new Date(selectedAdmin.createdAt).toLocaleDateString()}</p>
                  </div>
                </div>
              )}
            </DialogContent>
          </Dialog>

          {/* Edit Admin Modal */}
          <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
            <DialogContent className="max-w-md">
              <DialogHeader>
                <DialogTitle>Edit Admin</DialogTitle>
              </DialogHeader>
              {selectedAdmin && (
                <div className="space-y-4">
                  <div>
                    <Label htmlFor="editName">Name *</Label>
                    <Input
                      id="editName"
                      value={selectedAdmin.name}
                      onChange={(e) => setSelectedAdmin({ ...selectedAdmin, name: e.target.value })}
                      placeholder="Enter name"
                    />
                  </div>
                  <div>
                    <Label htmlFor="editEmail">Email *</Label>
                    <Input
                      id="editEmail"
                      type="email"
                      value={selectedAdmin.email}
                      onChange={(e) => setSelectedAdmin({ ...selectedAdmin, email: e.target.value })}
                      placeholder="Enter email"
                    />
                  </div>
                  <div>
                    <Label htmlFor="editPhone">Phone *</Label>
                    <Input
                      id="editPhone"
                      value={selectedAdmin.phone}
                      onChange={(e) => setSelectedAdmin({ ...selectedAdmin, phone: e.target.value })}
                      placeholder="Enter phone"
                    />
                  </div>
                  <Button onClick={handleSaveEdit} className="w-full">
                    Update Admin
                  </Button>
                </div>
              )}
            </DialogContent>
          </Dialog>

          {/* Delete Confirmation Modal */}
          <Dialog open={isDeleteModalOpen} onOpenChange={setIsDeleteModalOpen}>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Delete Admin</DialogTitle>
                <DialogDescription>
                  Are you sure you want to delete this admin? This action cannot be undone.
                </DialogDescription>
              </DialogHeader>
              <DialogFooter>
                <Button variant="outline" onClick={() => setIsDeleteModalOpen(false)}>
                  Cancel
                </Button>
                <Button variant="destructive" onClick={() => selectedAdmin && handleDeleteAdmin(selectedAdmin._id)}>
                  Delete
                </Button>
              </DialogFooter>
            </DialogContent>
          </Dialog>
        </div>
      </div>
    </DashboardLayout>
  )
}
