"use client"

import type React from "react"
import { useState, useEffect } from "react"
import { useSearchParams } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Database, DollarSign, Package, TrendingUp, Loader2, Calendar, MapPin } from "lucide-react"
import DashboardLayout from "@/components/dashboard-layout"
import { useBranches, useEmployees } from "@/hooks/use-dashboard-data"
import {
  useFormSubmission,
  transformSalesFormData,
  transformExpenseFormData,
  transformInventoryFormData,
} from "@/hooks/use-form-submissions"
import { useToast } from "@/hooks/use-toast"

export default function AddDataPage() {
  const [activeTab, setActiveTab] = useState("sales")
  const [expenseTab, setExpenseTab] = useState("cost-of-sales")
  const searchParams = useSearchParams()
  const { toast } = useToast()

  const [sharedFormData, setSharedFormData] = useState({
    date: new Date().toISOString().split("T")[0],
    branch: "",
  })

  const { branches, isLoading: branchesLoading } = useBranches()
  const { employees, isLoading: employeesLoading } = useEmployees(sharedFormData.branch)

  const dateFromPopup = searchParams.get("date")
  const branchFromPopup = searchParams.get("branch")
  const branchIdFromPopup = searchParams.get("branchId")
  const isFromPopup = !!(dateFromPopup || branchFromPopup)

  useEffect(() => {
    console.log("[v0] URL Parameters received:", { dateFromPopup, branchFromPopup, branchIdFromPopup })

    if (dateFromPopup || branchFromPopup) {
      const newFormData = {
        date: dateFromPopup || new Date().toISOString().split("T")[0],
        branch: branchIdFromPopup || branchFromPopup || "",
      }
      console.log("[v0] Setting shared form data from popup:", newFormData)
      setSharedFormData(newFormData)
    }
  }, [searchParams, dateFromPopup, branchFromPopup, branchIdFromPopup])

  useEffect(() => {
    if (branchFromPopup && branches && branches.length > 0 && !sharedFormData.branch) {
      // Find the branch by name or ID
      const foundBranch = branches.find(
        (b) => b.name.toLowerCase() === branchFromPopup.toLowerCase() || b._id === branchFromPopup,
      )
      if (foundBranch) {
        console.log("[v0] Auto-selecting branch from URL:", foundBranch)
        setSharedFormData((prev) => ({
          ...prev,
          branch: foundBranch._id,
        }))
      }
    }
  }, [branches, branchFromPopup, sharedFormData.branch])

  useEffect(() => {
    console.log("[v0] Shared form data updated:", sharedFormData)
    console.log("[v0] Employees data:", { employees, employeesLoading, branchId: sharedFormData.branch })
  }, [sharedFormData, employees, employeesLoading])

  const [formData, setFormData] = useState({
    sales: {
      cash_in_shop: "",
      cash_delivery: "",
      cash_wolt: "",
      cash_bolt: "",
      cash_pyszne_pi: "",
      cash_uber_eats: "",
      cash_glovo: "",
      card_in_shop: "",
      card_delivery: "",
      card_wolt: "",
      card_bolt: "",
      card_pyszne_pi: "",
      card_uber_eats: "",
      card_glovo: "",
      notes: "",
    },
    expense: {
      itemName: "",
      fakturaNo: "",
      notes: "",
      paymentMethod: "",
      netto: "",
      vat: "",
      brutto: "",
      paymentDate: "",
      employee: "",
      hourlyRate: "",
      totalHours: "",
      totalSalary: "",
      received: "",
      payable: "",
    },
    inventory: {
      type: "",
      kurczak: "",
      wolowina: "",
      kraftowe: "",
      majonez: "",
      ketchup: "",
      smabal: "",
      yogurt: "",
      pita_20_cm: "",
      pita_30_cm: "",
      gas: "",
      notes: "",
    },
  })

  const { isLoading, submitSalesData, submitExpenseData, submitInventoryData } = useFormSubmission()

  const handleBranchChange = (branchId: string) => {
    if (isFromPopup) return
    setSharedFormData((prev) => ({
      ...prev,
      branch: branchId,
    }))
  }

  const handleDateChange = (date: string) => {
    if (isFromPopup) return
    setSharedFormData((prev) => ({
      ...prev,
      date: date,
    }))
  }

  const handleInputChange = (tab: string, field: string, value: string) => {
    console.log("[v0] Input change:", { tab, field, value }) // Debug log
    setFormData((prev) => {
      const newFormData = {
        ...prev,
        [tab]: {
          ...prev[tab as keyof typeof prev],
          [field]: value,
        },
      }

      if (tab === "expense" && field === "employee") {
        const selectedEmployee = employees.find((emp) => emp._id === value)
        if (selectedEmployee) {
          newFormData.expense.hourlyRate = selectedEmployee.hourlyRate.toString()

          // Recalculate total salary if hours are already entered
          const totalHours = Number.parseFloat(newFormData.expense.totalHours) || 0
          if (totalHours > 0) {
            const totalSalary = (selectedEmployee.hourlyRate * totalHours).toFixed(2)
            newFormData.expense.totalSalary = totalSalary

            // Recalculate payable if received amount exists
            const received = Number.parseFloat(newFormData.expense.received) || 0
            const payable = (Number.parseFloat(totalSalary) - received).toFixed(2)
            newFormData.expense.payable = payable
          }
        }
      }

      if (tab === "expense" && (field === "hourlyRate" || field === "totalHours")) {
        const hourlyRate = Number.parseFloat(field === "hourlyRate" ? value : newFormData.expense.hourlyRate) || 0
        const totalHours = Number.parseFloat(field === "totalHours" ? value : newFormData.expense.totalHours) || 0
        const totalSalary = (hourlyRate * totalHours).toFixed(2)

        newFormData.expense.totalSalary = totalSalary

        // Recalculate payable if received amount exists
        const received = Number.parseFloat(newFormData.expense.received) || 0
        const payable = (Number.parseFloat(totalSalary) - received).toFixed(2)
        newFormData.expense.payable = payable
      }

      if (tab === "expense" && field === "received") {
        const totalSalary = Number.parseFloat(newFormData.expense.totalSalary) || 0
        const received = Number.parseFloat(value) || 0
        const payable = (totalSalary - received).toFixed(2)
        newFormData.expense.payable = payable
      }

      if (tab === "expense" && (field === "netto" || field === "vat")) {
        const netto = Number.parseFloat(field === "netto" ? value : newFormData.expense.netto) || 0
        const vat = Number.parseFloat(field === "vat" ? value : newFormData.expense.vat) || 0
        const brutto = (netto + vat).toFixed(2)
        newFormData.expense.brutto = brutto
      }

      return newFormData
    })
  }

  const handleSubmit = async (e: React.FormEvent, type: string) => {
    e.preventDefault()
    console.log("[v0] Form submission:", { type, sharedFormData, formData })

    if (!sharedFormData.branch) {
      toast({
        title: "Branch Required",
        description: "Please select a branch before submitting data.",
        variant: "destructive",
      })
      return
    }

    try {
      if (type === "sales" || type === "cash-sales" || type === "card-sales") {
        const salesData = transformSalesFormData(formData.sales, sharedFormData.branch, sharedFormData.date)
        console.log("[v0] Transformed sales data:", salesData)
        await submitSalesData(salesData)

        toast({
          title: "Sales Data Saved",
          description: "Sales data has been successfully saved to the database.",
        })

        // Reset form after successful submission
        setFormData((prev) => ({
          ...prev,
          sales: {
            cash_in_shop: "",
            cash_delivery: "",
            cash_wolt: "",
            cash_bolt: "",
            cash_pyszne_pi: "",
            cash_uber_eats: "",
            cash_glovo: "",
            card_in_shop: "",
            card_delivery: "",
            card_wolt: "",
            card_bolt: "",
            card_pyszne_pi: "",
            card_uber_eats: "",
            card_glovo: "",
            notes: "",
          },
        }))
      } else if (type === "expense" || type === "if-any-expense") {
        const expenseData = transformExpenseFormData(
          formData.expense,
          sharedFormData.branch,
          sharedFormData.date,
          expenseTab,
          formData.expense.itemName,
        )
        console.log("[v0] Transformed expense data:", expenseData)
        await submitExpenseData(expenseData)

        toast({
          title: "Expense Data Saved",
          description: `${expenseTab === "if-any" ? "IF ANY" : "Expense"} data has been successfully saved to the database.`,
        })

        // Reset form after successful submission
        setFormData((prev) => ({
          ...prev,
          expense: {
            itemName: "",
            fakturaNo: "",
            notes: "",
            paymentMethod: "",
            netto: "",
            vat: "",
            brutto: "",
            paymentDate: "",
            employee: "",
            hourlyRate: "",
            totalHours: "",
            totalSalary: "",
            received: "",
            payable: "",
          },
        }))
      } else if (type === "inventory-manage") {
        const inventoryData = transformInventoryFormData(
          formData.inventory,
          sharedFormData.branch,
          sharedFormData.date,
          formData.inventory.type,
        )
        console.log("[v0] Transformed inventory data:", inventoryData)
        await submitInventoryData(inventoryData)

        toast({
          title: "Inventory Data Saved",
          description: "Inventory data has been successfully saved to the database.",
        })

        // Reset form after successful submission
        setFormData((prev) => ({
          ...prev,
          inventory: {
            type: "",
            kurczak: "",
            wolowina: "",
            kraftowe: "",
            majonez: "",
            ketchup: "",
            smabal: "",
            yogurt: "",
            pita_20_cm: "",
            pita_30_cm: "",
            gas: "",
            notes: "",
          },
        }))
      }
    } catch (error) {
      console.error("[v0] Form submission error:", error)
      toast({
        title: "Submission Failed",
        description: "There was an error saving your data. Please try again.",
        variant: "destructive",
      })
    }
  }

  const expenseCategories = [
    {
      id: "cost-of-sales",
      label: "Cost Of Sales",
      items: ["Ali Baba", "Spec", "Naralny", "Coca-Cola", "Pepsi", "Trafin Oil", "Shop exp"],
    },
    {
      id: "utilities",
      label: "Utilities",
      items: ["Tauron", "Fortum", "Line Gas", "Cylinder Gas", "Internet Bill", "Garbage Bill"],
    },
    {
      id: "office-expense",
      label: "Office Expense",
      items: ["Accountant", "ZUS & PIT", "PFRON", "CIT", "TV", "PIT-11"],
    },
    { id: "delivery-costs", label: "Delivery Costs", items: ["Glovo", "Uber Eats", "Pyszne Pl", "Bolt", "Wolt"] },
    { id: "managing-expenses", label: "Managing Expenses", items: [] },
    { id: "shop-rents", label: "Shop Rents", items: [] },
    { id: "salaries", label: "Salaries", items: [] },
    { id: "fooding-expense", label: "Fooding Expense", items: [] },
    { id: "home-expenses", label: "Home Expenses", items: [] },
    { id: "salaries-calculations", label: "Salaries Calculations", items: [] },
    { id: "repair-maintenance", label: "Repair & Maintenance", items: [] },
    { id: "bank-charges", label: "Bank Charges", items: [] },
    { id: "others", label: "Others", items: [] },
    { id: "personal-withdraw", label: "Personal Withdraw", items: [] },
    { id: "if-any", label: "IF ANY", items: [] },
  ]

  return (
    <DashboardLayout>
      <div className="p-2 sm:p-4 lg:p-6">
        <div className="mx-auto max-w-6xl space-y-3 sm:space-y-4 lg:space-y-6">
          <div className="flex items-center gap-2 sm:gap-3">
            <div className="h-8 w-8 sm:h-10 sm:w-10 lg:h-12 lg:w-12 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center shadow-lg">
              <Database className="h-4 w-4 sm:h-5 sm:w-5 lg:h-6 lg:w-6 text-white" />
            </div>
            <h1 className="text-xl sm:text-2xl lg:text-3xl font-bold bg-gradient-to-r from-gray-900 to-gray-600 bg-clip-text text-transparent">
              Add Data
            </h1>
          </div>

          {isFromPopup ? (
            <Card className="border-0 shadow-lg bg-gradient-to-r from-green-50 to-blue-50 backdrop-blur-sm ring-1 ring-green-200/50">
              <CardContent className="px-3 sm:px-6 py-4">
                <div className="flex items-center gap-2 mb-3">
                  <div className="h-6 w-6 rounded-full bg-green-500 flex items-center justify-center">
                    <span className="text-white text-xs font-bold">✓</span>
                  </div>
                  <h3 className="text-sm font-semibold text-green-800">Selected from Add Data Popup</h3>
                </div>
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                  <div className="flex items-center gap-3 p-3 bg-white/60 rounded-lg border border-green-200">
                    <Calendar className="h-5 w-5 text-green-600" />
                    <div>
                      <Label className="text-xs font-medium text-gray-600">Selected Date</Label>
                      <p className="text-sm font-semibold text-gray-900">{sharedFormData.date}</p>
                    </div>
                  </div>
                  <div className="flex items-center gap-3 p-3 bg-white/60 rounded-lg border border-green-200">
                    <MapPin className="h-5 w-5 text-green-600" />
                    <div>
                      <Label className="text-xs font-medium text-gray-600">Selected Branch</Label>
                      <p className="text-sm font-semibold text-gray-900">
                        {branches?.find((b) => b._id === sharedFormData.branch)?.name ||
                          branchFromPopup ||
                          "Loading..."}
                      </p>
                    </div>
                  </div>
                </div>
                <p className="text-xs text-green-700 mt-3 text-center">
                  These values are locked and cannot be changed. They were selected from the Add Data popup.
                </p>
              </CardContent>
            </Card>
          ) : (
            <Card className="border-0 shadow-lg bg-white/80 backdrop-blur-sm ring-1 ring-gray-200/50">
              <CardContent className="px-3 sm:px-6 py-6">
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                  <div className="space-y-2">
                    <Label className="text-xs sm:text-sm font-medium text-gray-700">Select Date</Label>
                    <Input
                      type="date"
                      value={sharedFormData.date}
                      onChange={(e) => handleDateChange(e.target.value)}
                      className="h-10"
                    />
                    <div className="text-xs text-gray-500">Current date: {sharedFormData.date}</div>
                  </div>
                  <div className="space-y-2">
                    <Label className="text-xs sm:text-sm font-medium text-gray-700">Select Branch</Label>
                    {branchesLoading ? (
                      <div className="h-10 px-3 py-2 bg-muted rounded-md border text-xs sm:text-sm flex items-center">
                        <Loader2 className="h-4 w-4 animate-spin mr-2" />
                        Loading branches...
                      </div>
                    ) : (
                      <Select value={sharedFormData.branch} onValueChange={handleBranchChange}>
                        <SelectTrigger className="h-10">
                          <SelectValue placeholder="Select a branch" />
                        </SelectTrigger>
                        <SelectContent>
                          {branches?.map((branch) => (
                            <SelectItem key={branch._id} value={branch._id}>
                              {branch.name}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    )}
                    <div className="text-xs text-gray-500">
                      Current branch: {sharedFormData.branch || "None selected"}
                      {sharedFormData.branch && branches && (
                        <span className="ml-2 font-medium text-green-600">
                          ({branches.find((b) => b._id === sharedFormData.branch)?.name || "Unknown"})
                        </span>
                      )}
                    </div>
                  </div>
                </div>
                {!sharedFormData.branch && (
                  <div className="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-md">
                    <p className="text-sm text-yellow-800">Please select a branch to continue with data entry.</p>
                  </div>
                )}
              </CardContent>
            </Card>
          )}

          <div style={{ display: isFromPopup ? "none" : "block" }}>
            {/* This div will be hidden when values come from popup */}
          </div>

          {sharedFormData.branch && (
            <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-3 sm:space-y-4">
              <TabsList className="grid w-full grid-cols-3 h-auto p-1 bg-muted/50">
                <TabsTrigger
                  value="sales"
                  className="text-xs sm:text-sm py-2 px-1 sm:px-2 lg:px-4 flex flex-col sm:flex-row items-center gap-1 sm:gap-2"
                >
                  <DollarSign className="h-3 w-3 sm:h-4 sm:w-4" />
                  <span className="hidden xs:inline">Sales Data</span>
                  <span className="xs:hidden">Sales</span>
                </TabsTrigger>
                <TabsTrigger
                  value="expense"
                  className="text-xs sm:text-sm py-2 px-1 sm:px-2 lg:px-4 flex flex-col sm:flex-row items-center gap-1 sm:gap-2"
                >
                  <TrendingUp className="h-3 w-3 sm:h-4 sm:w-4" />
                  <span className="hidden xs:inline">Expense Data</span>
                  <span className="xs:hidden">Expense</span>
                </TabsTrigger>
                <TabsTrigger
                  value="inventory"
                  className="text-xs sm:text-sm py-2 px-1 sm:px-2 lg:px-4 flex flex-col sm:flex-row items-center gap-1 sm:gap-2"
                >
                  <Package className="h-3 w-3 sm:h-4 sm:w-4" />
                  <span className="hidden xs:inline">Inventory Data</span>
                  <span className="xs:hidden">Inventory</span>
                </TabsTrigger>
              </TabsList>

              {/* Sales Data Tab */}
              <TabsContent value="sales">
                <Card className="border-0 shadow-lg">
                  <CardHeader className="p-3 sm:p-6">
                    <CardTitle className="flex items-center gap-2 text-lg sm:text-xl">
                      <DollarSign className="h-4 w-4 sm:h-5 sm:w-5" />
                      Add Sales Data
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="p-3 sm:p-6">
                    <form onSubmit={(e) => handleSubmit(e, "sales")} className="space-y-6">
                      {/* Cash and Card Transaction Forms */}
                      <div className="grid grid-cols-1 xl:grid-cols-2 gap-4 sm:gap-6">
                        {/* Cash Transactions */}
                        <Card className="shadow-sm">
                          <CardHeader className="bg-primary text-primary-foreground p-3 sm:p-6">
                            <CardTitle className="flex items-center gap-2 text-base sm:text-lg">
                              <DollarSign className="h-4 w-4 sm:h-5 sm:w-5" />
                              Cash Transactions
                            </CardTitle>
                          </CardHeader>
                          <CardContent className="p-3 sm:p-6">
                            <div className="space-y-3 sm:space-y-4">
                              {/* Basic Cash Fields */}
                              <div className="space-y-3 sm:space-y-4">
                                <div className="space-y-2">
                                  <Label className="font-semibold flex items-center gap-2 text-sm sm:text-base">
                                    <Package className="h-3 w-3 sm:h-4 sm:w-4" />
                                    In Shop
                                    <span className="text-red-500">*</span>
                                  </Label>
                                  <div className="flex items-center gap-2">
                                    <span className="bg-muted px-2 sm:px-3 py-2 rounded-l-md border border-r-0 text-sm">
                                      zł
                                    </span>
                                    <Input
                                      type="number"
                                      name="cash_in_shop"
                                      placeholder="Enter amount"
                                      className="rounded-l-none border-l-0 flex-1"
                                      step="0.01"
                                      value={formData.sales.cash_in_shop}
                                      onChange={(e) => handleInputChange("sales", "cash_in_shop", e.target.value)}
                                      required
                                    />
                                  </div>
                                </div>

                                <div className="space-y-2">
                                  <Label className="font-semibold text-sm sm:text-base">
                                    Delivery
                                    <span className="text-red-500">*</span>
                                  </Label>
                                  <div className="flex items-center gap-2">
                                    <span className="bg-muted px-2 sm:px-3 py-2 rounded-l-md border border-r-0 text-sm">
                                      zł
                                    </span>
                                    <Input
                                      type="number"
                                      name="cash_delivery"
                                      placeholder="Enter amount"
                                      className="rounded-l-none border-l-0 flex-1"
                                      step="0.01"
                                      value={formData.sales.cash_delivery}
                                      onChange={(e) => handleInputChange("sales", "cash_delivery", e.target.value)}
                                      required
                                    />
                                  </div>
                                </div>
                              </div>

                              {/* Online Delivery Payments */}
                              <div className="border-t border-b py-3 sm:py-4 my-3 sm:my-4">
                                <h5 className="text-center text-primary font-semibold mb-3 sm:mb-4 text-sm sm:text-base">
                                  Online Delivery Payments
                                </h5>

                                <div className="space-y-3">
                                  {["Wolt", "Bolt", "Pyszne_pi", "Uber_eats", "Glovo"].map((platform) => (
                                    <div key={platform} className="space-y-2">
                                      <Label className="font-semibold text-sm sm:text-base">
                                        {platform.replace("_", " ")}
                                      </Label>
                                      <div className="flex items-center gap-2">
                                        <Input
                                          type="number"
                                          name={`cash_${platform.toLowerCase()}`}
                                          placeholder="0.00"
                                          step="0.01"
                                          className="rounded-r-none border-r-0 flex-1"
                                          value={formData.sales[`cash_${platform.toLowerCase()}`]}
                                          onChange={(e) =>
                                            handleInputChange("sales", `cash_${platform.toLowerCase()}`, e.target.value)
                                          }
                                        />
                                        <span className="bg-muted px-2 sm:px-3 py-2 rounded-r-md border border-l-0 text-sm">
                                          zł
                                        </span>
                                      </div>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          </CardContent>
                        </Card>

                        {/* Card Transactions */}
                        <Card className="shadow-sm">
                          <CardHeader className="bg-primary text-primary-foreground p-3 sm:p-6">
                            <CardTitle className="flex items-center gap-2 text-base sm:text-lg">
                              <DollarSign className="h-4 w-4 sm:h-5 sm:w-5" />
                              Card Transactions
                            </CardTitle>
                          </CardHeader>
                          <CardContent className="p-3 sm:p-6">
                            <div className="space-y-3 sm:space-y-4">
                              {/* Basic Card Fields */}
                              <div className="space-y-3 sm:space-y-4">
                                <div className="space-y-2">
                                  <Label className="font-semibold flex items-center gap-2 text-sm sm:text-base">
                                    <Package className="h-3 w-3 sm:h-4 sm:w-4" />
                                    In Shop
                                    <span className="text-red-500">*</span>
                                  </Label>
                                  <div className="flex items-center gap-2">
                                    <span className="bg-muted px-2 sm:px-3 py-2 rounded-l-md border border-r-0 text-sm">
                                      zł
                                    </span>
                                    <Input
                                      type="number"
                                      name="card_in_shop"
                                      placeholder="Enter amount"
                                      className="rounded-l-none border-l-0 flex-1"
                                      step="0.01"
                                      value={formData.sales.card_in_shop}
                                      onChange={(e) => handleInputChange("sales", "card_in_shop", e.target.value)}
                                      required
                                    />
                                  </div>
                                </div>

                                <div className="space-y-2">
                                  <Label className="font-semibold text-sm sm:text-base">
                                    Delivery
                                    <span className="text-red-500">*</span>
                                  </Label>
                                  <div className="flex items-center gap-2">
                                    <span className="bg-muted px-2 sm:px-3 py-2 rounded-l-md border border-r-0 text-sm">
                                      zł
                                    </span>
                                    <Input
                                      type="number"
                                      name="card_delivery"
                                      placeholder="Enter amount"
                                      className="rounded-l-none border-l-0 flex-1"
                                      step="0.01"
                                      value={formData.sales.card_delivery}
                                      onChange={(e) => handleInputChange("sales", "card_delivery", e.target.value)}
                                      required
                                    />
                                  </div>
                                </div>
                              </div>

                              {/* Online Delivery Payments */}
                              <div className="border-t border-b py-3 sm:py-4 my-3 sm:my-4">
                                <h5 className="text-center text-primary font-semibold mb-3 sm:mb-4 text-sm sm:text-base">
                                  Online Delivery Payments
                                </h5>

                                <div className="space-y-3">
                                  {["Wolt", "Bolt", "Pyszne_pi", "Uber_eats", "Glovo"].map((platform) => (
                                    <div key={platform} className="space-y-2">
                                      <Label className="font-semibold text-sm sm:text-base">
                                        {platform.replace("_", " ")}
                                      </Label>
                                      <div className="flex items-center gap-2">
                                        <Input
                                          type="number"
                                          name={`card_${platform.toLowerCase()}`}
                                          placeholder="0.00"
                                          step="0.01"
                                          className="rounded-r-none border-r-0 flex-1"
                                          value={formData.sales[`card_${platform.toLowerCase()}`]}
                                          onChange={(e) =>
                                            handleInputChange("sales", `card_${platform.toLowerCase()}`, e.target.value)
                                          }
                                        />
                                        <span className="bg-muted px-2 sm:px-3 py-2 rounded-r-md border border-l-0 text-sm">
                                          zł
                                        </span>
                                      </div>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          </CardContent>
                        </Card>
                      </div>

                      <div className="space-y-4">
                        <div className="space-y-2">
                          <Label htmlFor="sales-notes" className="text-sm font-medium">
                            Notes (Optional)
                          </Label>
                          <Input
                            id="sales-notes"
                            type="text"
                            placeholder="Add any additional notes..."
                            value={formData.sales.notes}
                            onChange={(e) => handleInputChange("sales", "notes", e.target.value)}
                          />
                        </div>

                        <Button
                          type="submit"
                          className="w-full bg-green-600 hover:bg-green-700 text-sm sm:text-base"
                          disabled={isLoading || !sharedFormData.branch}
                        >
                          {isLoading ? (
                            <>
                              <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                              Saving Sales Data...
                            </>
                          ) : (
                            <>
                              <DollarSign className="mr-2 h-4 w-4" />
                              Save Sales Data
                            </>
                          )}
                        </Button>
                      </div>
                    </form>
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Expense Data Tab */}
              <TabsContent value="expense">
                <Card className="border-0 shadow-lg">
                  <CardHeader className="p-3 sm:p-6">
                    <CardTitle className="flex items-center gap-2 text-lg sm:text-xl">
                      <TrendingUp className="h-4 w-4 sm:h-5 sm:w-5" />
                      Add Expense Data
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="p-3 sm:p-6">
                    <Tabs value={expenseTab} onValueChange={setExpenseTab} className="space-y-3 sm:space-y-4">
                      <div className="overflow-x-auto">
                        <TabsList className="flex w-max min-w-full gap-1 h-auto p-1 bg-muted/50">
                          {expenseCategories.map((category) => (
                            <TabsTrigger
                              key={category.id}
                              value={category.id}
                              className="text-xs px-2 py-2 whitespace-nowrap text-center font-medium transition-all duration-200 data-[state=active]:bg-primary data-[state=active]:text-primary-foreground hover:bg-muted-foreground/10 flex-shrink-0"
                            >
                              {category.label}
                            </TabsTrigger>
                          ))}
                        </TabsList>
                      </div>

                      {expenseCategories.map((category) => (
                        <TabsContent key={category.id} value={category.id}>
                          <Card className="border shadow-sm">
                            <CardHeader className="bg-primary text-primary-foreground p-3 sm:p-6">
                              <CardTitle className="text-base sm:text-lg">{category.label}</CardTitle>
                            </CardHeader>
                            <CardContent className="p-3 sm:p-6">
                              {category.id === "if-any" ? (
                                <form
                                  onSubmit={(e) => handleSubmit(e, "if-any-expense")}
                                  className="space-y-3 sm:space-y-4"
                                >
                                  <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                                    <div className="space-y-2">
                                      <Label htmlFor="if-any-amount" className="text-sm sm:text-base">
                                        If Any Amount
                                      </Label>
                                      <Input
                                        id="if-any-amount"
                                        type="number"
                                        placeholder="Enter amount"
                                        step="0.01"
                                        value={formData.expense.brutto}
                                        onChange={(e) => handleInputChange("expense", "brutto", e.target.value)}
                                        required
                                      />
                                    </div>
                                    <div className="space-y-2">
                                      <Label htmlFor="if-any-payment-date" className="text-sm sm:text-base">
                                        Payment Date
                                      </Label>
                                      <Input
                                        id="if-any-payment-date"
                                        type="date"
                                        value={formData.expense.paymentDate}
                                        onChange={(e) => handleInputChange("expense", "paymentDate", e.target.value)}
                                        required
                                      />
                                    </div>
                                  </div>
                                  <Button
                                    type="submit"
                                    className="w-full bg-green-600 hover:bg-green-700 mt-4 sm:mt-6 text-sm sm:text-base"
                                    disabled={isLoading || !sharedFormData.branch}
                                  >
                                    {isLoading ? (
                                      <>
                                        <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                                        Saving...
                                      </>
                                    ) : (
                                      "Create IF ANY Expense"
                                    )}
                                  </Button>
                                </form>
                              ) : (
                                <form onSubmit={(e) => handleSubmit(e, "expense")} className="space-y-3 sm:space-y-4">
                                  {/* Expense Form Fields */}
                                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4">
                                    {(category.id === "salaries" ||
                                      category.id === "fooding-expense" ||
                                      category.id === "managing-expenses" ||
                                      category.id === "salaries-calculations") && (
                                      <div className="space-y-2">
                                        <Label htmlFor="employee" className="text-sm sm:text-base">
                                          Select Employee
                                        </Label>
                                        {employeesLoading ? (
                                          <div className="h-10 px-3 py-2 bg-muted rounded-md border text-xs sm:text-sm flex items-center">
                                            <Loader2 className="h-4 w-4 animate-spin mr-2" />
                                            Loading employees...
                                          </div>
                                        ) : !sharedFormData.branch ? (
                                          <div className="h-10 px-3 py-2 bg-muted rounded-md border text-xs sm:text-sm flex items-center text-muted-foreground">
                                            Please select a branch first
                                          </div>
                                        ) : (
                                          <Select
                                            value={formData.expense.employee}
                                            onValueChange={(value) => {
                                              console.log("[v0] Employee selected:", value)
                                              handleInputChange("expense", "employee", value)
                                            }}
                                          >
                                            <SelectTrigger>
                                              <SelectValue
                                                placeholder={
                                                  employees.length === 0
                                                    ? "No employees found for this branch"
                                                    : "Select employee"
                                                }
                                              />
                                            </SelectTrigger>
                                            <SelectContent>
                                              {employees
                                                .filter((emp) => emp.status === "active")
                                                .map((employee) => (
                                                  <SelectItem key={employee._id} value={employee._id}>
                                                    <div className="flex flex-col">
                                                      <span className="font-medium">{employee.name}</span>
                                                      <span className="text-xs text-muted-foreground">
                                                        {employee.position} • {employee.hourlyRate} zł/hr
                                                      </span>
                                                    </div>
                                                  </SelectItem>
                                                ))}
                                            </SelectContent>
                                          </Select>
                                        )}
                                        {employees.length === 0 && !employeesLoading && sharedFormData.branch && (
                                          <p className="text-xs text-muted-foreground">
                                            No active employees found for the selected branch.
                                            <br />
                                            <span
                                              className="text-blue-600 underline cursor-pointer"
                                              onClick={() => window.open("/employees", "_blank")}
                                            >
                                              Add employees first
                                            </span>
                                          </p>
                                        )}
                                      </div>
                                    )}

                                    {category.id === "salaries-calculations" && (
                                      <>
                                        <div className="space-y-2">
                                          <Label htmlFor="hours-rate" className="text-sm sm:text-base">
                                            Hours Rate
                                          </Label>
                                          <Input
                                            id="hours-rate"
                                            type="number"
                                            placeholder="Amount"
                                            step="0.01"
                                            value={formData.expense.hourlyRate || ""}
                                            onChange={(e) => handleInputChange("expense", "hourlyRate", e.target.value)}
                                            readOnly={!!formData.expense.employee}
                                            className={formData.expense.employee ? "bg-muted" : ""}
                                          />
                                          {formData.expense.employee && (
                                            <p className="text-xs text-muted-foreground">
                                              Rate automatically set from selected employee
                                            </p>
                                          )}
                                        </div>

                                        <div className="space-y-2">
                                          <Label htmlFor="total-hours" className="text-sm sm:text-base">
                                            Total Hours
                                          </Label>
                                          <Input
                                            id="total-hours"
                                            type="number"
                                            placeholder="Hours"
                                            step="0.01"
                                            value={formData.expense.totalHours || ""}
                                            onChange={(e) => handleInputChange("expense", "totalHours", e.target.value)}
                                          />
                                        </div>

                                        <div className="space-y-2">
                                          <Label htmlFor="total-salary" className="text-sm sm:text-base">
                                            Total Salary
                                          </Label>
                                          <Input
                                            id="total-salary"
                                            type="number"
                                            placeholder="Total Salary"
                                            step="0.01"
                                            value={formData.expense.totalSalary || ""}
                                            onChange={(e) =>
                                              handleInputChange("expense", "totalSalary", e.target.value)
                                            }
                                            readOnly
                                          />
                                        </div>

                                        <div className="space-y-2">
                                          <Label htmlFor="received" className="text-sm sm:text-base">
                                            Received
                                          </Label>
                                          <Input
                                            id="received"
                                            type="number"
                                            placeholder="Amount"
                                            step="0.01"
                                            value={formData.expense.received || ""}
                                            onChange={(e) => handleInputChange("expense", "received", e.target.value)}
                                          />
                                        </div>

                                        <div className="space-y-2">
                                          <Label htmlFor="payable" className="text-sm sm:text-base">
                                            Payable
                                          </Label>
                                          <Input
                                            id="payable"
                                            type="number"
                                            placeholder="Amount"
                                            step="0.01"
                                            value={formData.expense.payable || ""}
                                            onChange={(e) => handleInputChange("expense", "payable", e.target.value)}
                                            readOnly
                                          />
                                        </div>
                                      </>
                                    )}

                                    {category.id === "home-expenses" && (
                                      <div className="space-y-2">
                                        <Label htmlFor="home-expense" className="text-sm sm:text-base">
                                          Select Home Expense
                                        </Label>
                                        <Select
                                          value={formData.expense.itemName}
                                          onValueChange={(value) => handleInputChange("expense", "itemName", value)}
                                        >
                                          <SelectTrigger>
                                            <SelectValue placeholder="Select home expense" />
                                          </SelectTrigger>
                                          <SelectContent>
                                            <SelectItem value="rent">Rent</SelectItem>
                                            <SelectItem value="utilities">Utilities</SelectItem>
                                            <SelectItem value="groceries">Groceries</SelectItem>
                                            <SelectItem value="maintenance">Maintenance</SelectItem>
                                            <SelectItem value="insurance">Insurance</SelectItem>
                                          </SelectContent>
                                        </Select>
                                      </div>
                                    )}

                                    {category.id === "repair-maintenance" && (
                                      <div className="space-y-2">
                                        <Label htmlFor="repair-maintenance" className="text-sm sm:text-base">
                                          Select Repair & Maintenance
                                        </Label>
                                        <Select
                                          value={formData.expense.itemName}
                                          onValueChange={(value) => handleInputChange("expense", "itemName", value)}
                                        >
                                          <SelectTrigger>
                                            <SelectValue placeholder="Select repair & maintenance" />
                                          </SelectTrigger>
                                          <SelectContent>
                                            <SelectItem value="equipment-repair">Equipment Repair</SelectItem>
                                            <SelectItem value="building-maintenance">Building Maintenance</SelectItem>
                                            <SelectItem value="plumbing">Plumbing</SelectItem>
                                            <SelectItem value="electrical">Electrical</SelectItem>
                                            <SelectItem value="hvac">HVAC</SelectItem>
                                          </SelectContent>
                                        </Select>
                                      </div>
                                    )}

                                    {category.id === "personal-withdraw" && (
                                      <div className="space-y-2">
                                        <Label htmlFor="personal-withdraw" className="text-sm sm:text-base">
                                          Select Personal Withdraw
                                        </Label>
                                        <Select
                                          value={formData.expense.itemName}
                                          onValueChange={(value) => handleInputChange("expense", "itemName", value)}
                                        >
                                          <SelectTrigger>
                                            <SelectValue placeholder="Select personal withdraw" />
                                          </SelectTrigger>
                                          <SelectContent>
                                            <SelectItem value="owner-draw">Owner Draw</SelectItem>
                                            <SelectItem value="personal-expenses">Personal Expenses</SelectItem>
                                            <SelectItem value="family-allowance">Family Allowance</SelectItem>
                                            <SelectItem value="personal-loan">Personal Loan</SelectItem>
                                            <SelectItem value="dividend">Dividend</SelectItem>
                                          </SelectContent>
                                        </Select>
                                      </div>
                                    )}

                                    {category.items.length > 0 && (
                                      <div className="space-y-2">
                                        <Label htmlFor="item-name" className="text-sm sm:text-base">
                                          Select {category.label}
                                        </Label>
                                        <Select
                                          value={formData.expense.itemName}
                                          onValueChange={(value) => handleInputChange("expense", "itemName", value)}
                                        >
                                          <SelectTrigger>
                                            <SelectValue placeholder={`Select ${category.label.toLowerCase()}`} />
                                          </SelectTrigger>
                                          <SelectContent>
                                            {category.items.map((item) => (
                                              <SelectItem key={item} value={item.toLowerCase().replace(/\s+/g, "-")}>
                                                {item}
                                              </SelectItem>
                                            ))}
                                          </SelectContent>
                                        </Select>
                                      </div>
                                    )}

                                    <div className="space-y-2">
                                      <Label htmlFor="faktura-no" className="text-sm sm:text-base">
                                        Faktura No.
                                      </Label>
                                      <Input
                                        id="faktura-no"
                                        type="text"
                                        placeholder="Faktura No"
                                        value={formData.expense.fakturaNo}
                                        onChange={(e) => handleInputChange("expense", "fakturaNo", e.target.value)}
                                      />
                                    </div>

                                    <div className="space-y-2">
                                      <Label htmlFor="notes" className="text-sm sm:text-base">
                                        Notes
                                      </Label>
                                      <Input
                                        id="notes"
                                        type="text"
                                        placeholder="Notes"
                                        value={formData.expense.notes}
                                        onChange={(e) => handleInputChange("expense", "notes", e.target.value)}
                                      />
                                    </div>

                                    <div className="space-y-2">
                                      <Label htmlFor="payment-method" className="text-sm sm:text-base">
                                        Payment Method
                                      </Label>
                                      <Select
                                        value={formData.expense.paymentMethod}
                                        onValueChange={(value) => handleInputChange("expense", "paymentMethod", value)}
                                      >
                                        <SelectTrigger>
                                          <SelectValue placeholder="Select payment method" />
                                        </SelectTrigger>
                                        <SelectContent>
                                          <SelectItem value="cash">Cash</SelectItem>
                                          <SelectItem value="card">Card</SelectItem>
                                        </SelectContent>
                                      </Select>
                                    </div>

                                    <div className="space-y-2">
                                      <Label htmlFor="netto" className="text-sm sm:text-base">
                                        Netto
                                      </Label>
                                      <Input
                                        id="netto"
                                        type="number"
                                        placeholder="Netto"
                                        value={formData.expense.netto}
                                        onChange={(e) => handleInputChange("expense", "netto", e.target.value)}
                                      />
                                    </div>

                                    <div className="space-y-2">
                                      <Label htmlFor="vat" className="text-sm sm:text-base">
                                        VAT
                                      </Label>
                                      <Input
                                        id="vat"
                                        type="number"
                                        placeholder="VAT"
                                        value={formData.expense.vat}
                                        onChange={(e) => handleInputChange("expense", "vat", e.target.value)}
                                      />
                                    </div>

                                    <div className="space-y-2">
                                      <Label htmlFor="brutto" className="text-sm sm:text-base">
                                        Brutto
                                      </Label>
                                      <Input
                                        id="brutto"
                                        type="number"
                                        placeholder="Brutto"
                                        value={formData.expense.brutto}
                                        onChange={(e) => handleInputChange("expense", "brutto", e.target.value)}
                                        readOnly
                                      />
                                    </div>

                                    <div className="space-y-2">
                                      <Label htmlFor="attach-file" className="text-sm sm:text-base">
                                        Attach File
                                      </Label>
                                      <Input id="attach-file" type="file" accept="image/*,.pdf" />
                                    </div>

                                    <div className="space-y-2">
                                      <Label htmlFor="payment-date" className="text-sm sm:text-base">
                                        Payment Date
                                      </Label>
                                      <Input
                                        id="payment-date"
                                        type="date"
                                        value={formData.expense.paymentDate}
                                        onChange={(e) => handleInputChange("expense", "paymentDate", e.target.value)}
                                      />
                                    </div>
                                  </div>

                                  <Button
                                    type="submit"
                                    className="w-full bg-green-600 hover:bg-green-700 mt-4 sm:mt-6 text-sm sm:text-base"
                                    disabled={isLoading || !sharedFormData.branch}
                                  >
                                    {isLoading ? (
                                      <>
                                        <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                                        Saving...
                                      </>
                                    ) : (
                                      `Create ${category.label}`
                                    )}
                                  </Button>
                                </form>
                              )}
                            </CardContent>
                          </Card>
                        </TabsContent>
                      ))}
                    </Tabs>
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Inventory Data Tab */}
              <TabsContent value="inventory">
                <Card className="border-0 shadow-lg">
                  <CardHeader className="p-3 sm:p-6">
                    <CardTitle className="flex items-center gap-2 text-lg sm:text-xl">
                      <Package className="h-4 w-4 sm:h-5 sm:w-5" />
                      Add Inventory Data
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="p-3 sm:p-6">
                    {/* Branch, Date, and Type Selection */}
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4 mb-4 sm:mb-6">
                      <div className="space-y-2">
                        <Label htmlFor="inventory-type" className="text-sm sm:text-base">
                          Select Type
                        </Label>
                        <Select
                          value={formData.inventory.type || ""}
                          onValueChange={(value) => handleInputChange("inventory", "type", value)}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select type" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="sub">Daily Consumption</SelectItem>
                            <SelectItem value="add">Add Inventory</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                    </div>

                    {/* Inventory Management Cards */}
                    <div className="grid grid-cols-1 xl:grid-cols-2 gap-4 sm:gap-6">
                      {/* Inventory Manage Card */}
                      <Card className="shadow-sm">
                        <CardHeader className="bg-primary text-primary-foreground p-3 sm:p-6">
                          <CardTitle className="flex items-center gap-2 text-base sm:text-lg">
                            <Package className="h-4 w-4 sm:h-5 sm:w-5" />
                            Inventory Manage
                          </CardTitle>
                        </CardHeader>
                        <CardContent className="p-3 sm:p-6">
                          <form
                            onSubmit={(e) => handleSubmit(e, "inventory-manage")}
                            className="space-y-3 sm:space-y-4"
                          >
                            {/* Food Items */}
                            {[
                              { name: "Kurczak", unit: "KG", placeholder: "Enter kg" },
                              { name: "Wolowina", unit: "KG", placeholder: "Enter kg" },
                              { name: "Kraftowe", unit: "KG", placeholder: "Enter kg" },
                              { name: "Majonez", unit: "KG", placeholder: "Enter kg" },
                              { name: "Ketchup", unit: "KG", placeholder: "Enter kg" },
                              { name: "Smabal", unit: "KG", placeholder: "Enter kg" },
                              { name: "Yogurt", unit: "KG", placeholder: "Enter kg" },
                              { name: "Pita 20 cm", unit: "PC", placeholder: "Enter pc" },
                              { name: "Pita 30 cm", unit: "PC", placeholder: "Enter pc" },
                              { name: "Gas", unit: "PC", placeholder: "Enter pc" },
                            ].map((item) => (
                              /* Mobile-first inventory item layout */
                              <div key={item.name} className="space-y-2">
                                <Label className="font-semibold flex items-center gap-2 text-sm sm:text-base">
                                  <Package className="h-3 w-3 sm:h-4 sm:w-4 flex-shrink-0" />
                                  {item.name}
                                  <span className="text-red-500">*</span>
                                </Label>
                                <div className="flex items-center gap-2">
                                  <Input
                                    type="number"
                                    name={item.name.toLowerCase().replace(/\s+/g, "_")}
                                    placeholder={item.placeholder}
                                    step="0.01"
                                    className="rounded-r-none border-r-0 flex-1"
                                    value={formData.inventory[item.name.toLowerCase().replace(/\s+/g, "_")]}
                                    onChange={(e) =>
                                      handleInputChange(
                                        "inventory",
                                        item.name.toLowerCase().replace(/\s+/g, "_"),
                                        e.target.value,
                                      )
                                    }
                                    required
                                  />
                                  <span className="bg-muted px-2 sm:px-3 py-2 rounded-r-md border border-l-0 text-sm font-medium">
                                    {item.unit}
                                  </span>
                                </div>
                              </div>
                            ))}

                            <div className="space-y-2">
                              <Label htmlFor="inventory-notes" className="text-sm font-medium">
                                Notes (Optional)
                              </Label>
                              <Input
                                id="inventory-notes"
                                type="text"
                                placeholder="Add any additional notes..."
                                value={formData.inventory.notes}
                                onChange={(e) => handleInputChange("inventory", "notes", e.target.value)}
                              />
                            </div>

                            <Button
                              type="submit"
                              className="w-full bg-green-600 hover:bg-green-700 mt-4 sm:mt-6 text-sm sm:text-base"
                              disabled={isLoading || !sharedFormData.branch || !formData.inventory.type}
                            >
                              {isLoading ? (
                                <>
                                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                                  Saving Inventory...
                                </>
                              ) : (
                                <>
                                  <Package className="mr-2 h-3 w-3 sm:h-4 sm:w-4" />
                                  Create Inventory
                                </>
                              )}
                            </Button>
                          </form>
                        </CardContent>
                      </Card>

                      {/* Stock Inventory Card */}
                      <Card className="shadow-sm">
                        <CardHeader className="bg-primary text-primary-foreground p-3 sm:p-6">
                          <CardTitle className="flex items-center gap-2 text-base sm:text-lg">
                            <Package className="h-4 w-4 sm:h-5 sm:w-5" />
                            Stock Inventory
                          </CardTitle>
                        </CardHeader>
                        <CardContent className="p-3 sm:p-6">
                          {/* Stock Items - Read Only Display */}
                          <div className="space-y-3 sm:space-y-4">
                            {[
                              { name: "Kurczak", unit: "KG", stockValue: "25.50" },
                              { name: "Wolowina", unit: "KG", stockValue: "18.75" },
                              { name: "Kraftowe", unit: "KG", stockValue: "12.00" },
                              { name: "Majonez", unit: "KG", stockValue: "8.25" },
                              { name: "Ketchup", unit: "KG", stockValue: "6.50" },
                              { name: "Smabal", unit: "KG", stockValue: "4.75" },
                              { name: "Yogurt", unit: "KG", stockValue: "15.00" },
                              { name: "Pita 20 cm", unit: "PC", stockValue: "150" },
                              { name: "Pita 30 cm", unit: "PC", stockValue: "200" },
                              { name: "Gas", unit: "PC", stockValue: "3" },
                            ].map((item) => (
                              /* Mobile-first stock item display */
                              <div
                                key={item.name}
                                className="flex flex-col sm:flex-row sm:items-center sm:justify-between p-3 bg-muted/30 rounded-lg gap-2 sm:gap-0"
                              >
                                <Label className="font-semibold flex items-center gap-2 text-sm sm:text-base">
                                  <Package className="h-3 w-3 sm:h-4 sm:w-4 flex-shrink-0" />
                                  {item.name}
                                </Label>
                                <div className="flex items-center gap-2 sm:w-40">
                                  <div className="bg-background px-2 sm:px-3 py-2 rounded-l-md border border-r-0 flex-1 text-right font-mono text-sm">
                                    {item.stockValue}
                                  </div>
                                  <span className="bg-primary text-primary-foreground px-2 sm:px-3 py-2 rounded-r-md border border-l-0 text-sm font-medium">
                                    {item.unit}
                                  </span>
                                </div>
                              </div>
                            ))}
                          </div>
                        </CardContent>
                      </Card>
                    </div>
                  </CardContent>
                </Card>
              </TabsContent>
            </Tabs>
          )}
        </div>
      </div>
    </DashboardLayout>
  )
}
